<?php
/**
 * Node Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
namespace Mtc\Core;

use DB;
use Illuminate\Database\Eloquent\Model;
use Laravel\Scout\Searchable;
use Mtc\Core\CustomFields\CustomGroup;
use Mtc\Core\Taxonomy;
use Nicolaslopezj\Searchable\SearchableTrait;

/**
 * The 'Node' is a cornerstone model that can connect to multiple different
 * models.
 *
 * It's mainly used to define a common set of fields, store a
 * unique slug which could later be used for access, and open access to
 * Custom Fields.
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Node extends Model
{
    use Searchable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'slug',
        'description',
        'status',
        'visibility',
        'group_id',
    ];

    /**
     * The array which is serialized by Laravel Scout to provide text based
     * searching.
     *
     * @return array
     */
    public function toSearchableArray()
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            // 'description' => $this->description,
            'nodeable_type' => $this->nodeable_type
        ];
    }

    /**
     * Register a custom save event to create slugs and generate URLs.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::saving(
            function ($node) {
                if (empty($node->slug)) {
                    $node->createSlug(str_slug($node->title));
                } else {
                    $node->createSlug($node->slug);
                }

                if (!empty($node->nodeable)
                    && method_exists($node->nodeable, 'getUrl')
                ) {
                    $node->url = $node->nodeable->getUrl(false, $node);
                }
            }
        );
    }

    /**
     * Get a list of all owning nodeable models.
     *
     * @return \Illuminate\Database\Relations\MorphTo
     */
    public function nodeable()
    {
        return $this->morphTo();
    }

    /**
     * Retrive information about a custom group associated with this Node.
     *
     * @return \Illuminate\Database\Relations\BelongsTo
     */
    public function group()
    {
        return $this->belongsTo(CustomGroup::class, 'group_id');
    }

    /**
     * Retrieve the custom data for the current node.
     *
     * @return [type] [description]
     */
    public function getGroupDataAttribute()
    {
        $data = DB::table($this->group->table_name)
            ->where('node_id', '=', $this->id)
            ->first();

        if (empty($data)) {
            return $data;
        }

        return collect($data)->map(
            function ($item) {
                $data = @unserialize($item);
                if ($data !== false) {
                    return $data;
                }

                return $item;
            }
        );
    }

    /**
     * Save custom data for the current node.
     *
     * @param array $data Incoming data
     *
     * @return boolean
     */
    public function saveGroupData(array $data)
    {
        $item = $this->group_data;
        $query = DB::table($this->group->table_name);
        $data['node_id'] = $this->id;

        // Sync taxonomies directly to node_taxonomy for better filter performance
        $this->taxonomies()->sync(
            collect($data)
                ->reject(
                    function ($item, $key) {
                        // Only return values that exist in a list
                        $field = $this->group
                            ->fields
                            ->where('column_name', $key)
                            ->where('list_id', '>', 0)
                            ->first();
                        if (empty($field) || empty($field->list_id)) {
                            return true;
                        }

                        return false;
                    }
                )
                ->flatten()
                ->toArray()
        );

        // Ensure data is serialized before going in.
        $data = collect($data)
            ->map(
                function ($item) {
                    if (is_array($item)) {
                        $item = serialize($item);
                    }

                    return $item;
                }
            )
            ->toArray();

        if ($item) {
            return $query->where('node_id', '=', $this->id)
                ->update($data);
        } else {
            return $query->insert($data);
        }
    }

    /**
     * Create a unique slug for the current node, set in $this->slug
     *
     * @param string $slug Slug to match against.
     *
     * @return void
     */
    public function createSlug($slug)
    {
        $base = $slug;
        $index = 1;

        // Check to see if a slug exists. If it does, then we start
        // adding an index until it's unique. E.g. hats, hats-1, hats-2.
        while ($this->slugExists($slug) == false) {
            $slug = $base . '-' . $index;
            $index++;
        }

        $this->slug = $slug;
    }

    /**
     * Determine if a slug already exists matching the provided string, except
     * for the current node.
     *
     * @param string $slug Slug to compare
     *
     * @return bool
     */
    public function slugExists($slug)
    {
        return self::whereSlug($slug)
            ->where('id', '!=', $this->id)
            ->count() == 0;
    }

    /**
     * Get the related taxonomies through the node_taxonomy pivot table
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function taxonomies()
    {
        return $this->belongsToMany(Taxonomy::class);
    }

    /**
     * Provide a public scope to ensure that the node is published and
     * available to the public.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePublic($query)
    {
        return $query->whereStatus('published')
            ->whereVisibility('public');
    }
}
