<?php
/**
 * Builder
 *
 * Handles the generation of admin area browse pages.
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Admin;

use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;

/**
 * Used to generate index pages for the admin area.
 *
 * @category Mtc\Core\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Builder
{
    /**
     * The name of the resource. Used for route generating.
     *
     * @var string
     */
    private $resource_name = '';

    /**
     * The title of the resource.
     *
     * @var string
     */
    public $title = '';

    /**
     * A model's Query Builder for us to paginate with
     *
     * @var null|QueryBuilder
     */
    private $query = null;

    /**
     * The column names / titles on the table
     *
     * @var array
     */
    private $columns = [];

    /**
     * Any custom data to be passed to the view (closures only)
     *
     * @var array
     */
    private $data = [];

    /**
     * Store any additional route parameters if required.
     *
     * @var array
     */
    private $route_params = [];

    /**
     * A permission required to delete an item within the list.
     *
     * @var string
     */
    private $delete_permission = '';

    /**
     * Build up the Builder instance.
     *
     * @param string       $resource_name The name to the current route
     * @param QueryBuilder $query         The query builder of the result model
     *
     * @return void
     */
    public function __construct($resource_name, QueryBuilder $query)
    {
        $this->resource_name = $resource_name;
        $this->query = $query;

        $title = explode('.', $resource_name);
        $this->title = end($title);
    }

    /**
     * The columns that we want on the table.
     *
     * @param array $columns key = attribute name, value = title.
     *
     * @return self
     */
    public function columns(array $columns)
    {
        $this->columns = $columns;
        return $this;
    }

    /**
     * Store additional route parameters if required.
     *
     * @param array $params array
     *
     * @return self
     */
    public function routeParams(array $params)
    {
        $this->route_params = $params;
        return $this;
    }

    /**
     * Pass through any custom data to the table, passed as closures.
     *
     * @param Closure[] $data An array of closures
     *
     * @return self
     */
    public function data(array $data)
    {
        $this->data = collect($data)
            ->filter(
                function ($closure) {
                    return is_object($closure) &&
                       $closure instanceof Closure;
                }
            )->toArray();
        return $this;
    }

    /**
     * If no columns are provided, auto-detect the table columns by the
     * model's attributes.
     *
     * @param Model $model Eloquent Model
     *
     * @return self
     */
    public function setBaseColumns(Model $model)
    {
        // Get the keys of the current model in a collection
        $keys = collect(array_keys($model->toArray()));

        // Convert to a translated 2d array
        $this->columns = $keys->map(
            function ($key) {
                return [
                'key' => $key,
                'name' => trans("fields.{$key}"),
                ];
            }
        )->keyBy('key')
        ->map(
            function ($key) {
                return $key['key'];
            }
        )->toArray();

        return $this;
    }

    /**
     * Set Entrust permission that is required to delete records on this view.
     *
     * @param string $permission Key from the permissions table.
     */
    public function setDeletePermission(string $permission)
    {
        $this->delete_permission = $permission;
        return $this;
    }

    /**
     * Render a view for the current Admin page.
     *
     * @return Illumnate\Http\Response
     */
    public function view()
    {
        $data = $this->query->paginate();

        // Bring in any custom data.
        if (!empty($this->data)) {
            $data->map(
                function ($item, $key) {
                    foreach ($this->data as $key => $closure) {
                        $item[$key] = $closure($item);
                    }
                    return $item;
                }
            );
        };

        if (empty($this->columns) && !$data->isEmpty()) {
            $this->setBaseColumns($data->first());
        }

        $views = [
            'head' => event("core.builder.{$this->resource_name}.head"),
            'foot' => event("core.builder.{$this->resource_name}.foot"),
            'buttons' => event("core.builder.{$this->resource_name}.buttons"),
        ];

        return view('core::admin.builder.index')->with(
            [
                'title' => $this->title,
                'name' => $this->resource_name,
                'data' => $data,
                'columns' => $this->columns,
                'route_params' => $this->route_params,
                'views' => $views,
                'delete_permission' => $this->delete_permission,
            ]
        );
    }
}
