<?php
/**
 * ItemBuilder - Handles the generation of admin area item pages.
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Admin;

use Illuminate\Database\Eloquent\Model;

/**
 * ItemBuilder class
 *
 * @category Mtc\Core\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ItemBuilder
{
    /**
     * The name of the base route for this page (e.g. core.admin.users)
     *
     * @var string
     */
    protected $resource_name = '';
    /**
     * The model in which we are running for this page
     *
     * @var null|Model
     */
    protected $item = null;
    /**
     * The Node which is assigned to the above $item
     *
     * @var null|Mtc\Core\Node
     */
    protected $node = null;
    /**
     * Store any additional parameters required to generate routes.
     *
     * @var array
     */
    protected $route_params = [];

    /**
     * Set the defaults for resource, Model, and Node.
     *
     * @param string             $resource_name The path to the route resource
     * @param Model              $item          The Eloquent model
     * @param null|Mtc\Core\Node $node          The associated Node (or null)
     *
     * @return void
     */
    public function __construct($resource_name, Model $item, $node = null)
    {
        $this->resource_name = $resource_name;
        $this->item = $item;

        if (class_basename($node) == 'Node') {
            $this->node = $node;
        }
    }

    /**
     * Store additional route parameters if required.
     *
     * These may be required if your model is built off of another class
     * (e.g. A product and it's images, where we need to know the related
     * product)
     *
     * @param array $params Additional Route Params
     *
     * @return self
     */
    public function routeParams(array $params)
    {
        $this->route_params = $params;
        return $this;
    }

    /**
     * Render the view with an optional sub-view.
     *
     * @param string $view Optional additional view to render inside template.
     *
     * @return \Illuminate\View\View
     */
    public function view($view = '')
    {
        // Set the default action to create an item
        $action = call_user_func_array(
            'route', [
            "{$this->resource_name}.store",
            $this->route_params
            ]
        );

        $method = 'POST';
        // If the model exists, then we are updating
        if ($this->item->exists) {
            $action = call_user_func_array(
                'route', [
                "{$this->resource_name}.update",
                array_merge($this->route_params, [$this->item->id])
                ]
            );

            $method = 'PUT';
        }

        return view('core::admin.builder.item')
            ->with(
                [
                'view' => $view,
                'name' => $this->resource_name,
                'item' => $this->item,
                'node' => $this->node,
                'form_action' => $action,
                'form_method' => $method,
                'route_params' => $this->route_params,
                ]
            );
    }

    /**
     * Render any tabs associated with this item.
     *
     * @param string     $view Include an additional view in render.
     * @param null|array $data Any data to be included, including 'buttons'.
     *                          Buttons array stores the following in array for
     *                          each tab:
     *                          - label
     *                          - href
     *                          - class
     *
     * @return \Illuminate\View\View
     */
    public function tab($view, $data = null)
    {
        return view('core::admin.builder.tab')
            ->with(
                array_merge(
                    [
                    'view' => $view,
                    'item' => $this->item,
                    'node' => $this->node,
                    'name' => $this->resource_name,
                    'form_action' => '',
                    'form_method' => '',
                    ], (array)$data
                )
            );
    }
}
