<?php
/**
 * Migration Builer for CustomFields
 *
 * PHP Version 7
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\CustomFields;

use Illuminate\Support\Collection;

/**
 * Create and Run a Migration. Used within Custom Fields to create
 * a table for each group and their fields.
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Migration
{
    /**
     * The migration mode (edit/create/drop)
     *
     * @var string
     */
    protected $mode = '';
    /**
     * The CustomGroup containing information about the table/fields.
     *
     * @var null|CustomGroup
     */
    protected $group = null;

    /**
     * The sections which will be replaced within the stubbed migrations.
     *
     * @var array
     */
    protected $sections = [
        'migration_name' => '',
        'table_name' => '',
        'rename' => '',
        'up' => [],
        'down' => [],
    ];

    /**
     * Set the base mode, group and determine if a table has been renamed.
     *
     * @param CustomGroup $group The custom group this migration is for
     */
    public function __construct(CustomGroup $group)
    {
        $this->group = $group;
        $this->mode = $group->exists ? 'edit' : 'create';

        // If exists and is dirty, the table name has changed and will
        // be renamed to match new name.
        if ($group->exists && $group->isDirty('table_name')) {
            $original = $group->getOriginal('table_name');
            $this->sections['rename'] = "Schema::rename('{$original}', '{$group->table_name}');";
        }
    }

    /**
     * Add fields if they don't exist or have been renamed.
     *
     * @param CustomField $field The CustomField being added to the migration
     *
     * @return self
     */
    public function addField(CustomField $field)
    {
        if ($field->exists === false) {
            // As field doesn't exist, create it.
            $this->sections['up'][] = "\$table->string('{$field->column_name}')->default(null)->nullable()->index();";
            $this->sections['down'][] = "\$table->dropColumn('{$field->column_name}');";
        } elseif ($field->isDirty('column_name')) {
            // Field has been renamed
            $original = $field->getOriginal('column_name');
            $this->sections['up'][] = "\$table->renameColumn('{$old}', '{$field->column_name}');";
            $this->sections['down'][] = "\$table->renameColumn('{$field->column_name}', '{$old}');";
        }

        return $this;
    }

    /**
     * Remove fields with the current migration by providing a
     * collection of column names.
     *
     * @param Collection $columns strings of column names
     *
     * @return self
     */
    public function removeFields(Collection $columns)
    {
        $columns->each(
            function ($column) {
                $this->sections['up'][] = "\$table->dropColumn('{$column}');";
                $this->sections['down'][] = "\$table->string('{$column}')->default(null)->nullable()->index();";
            }
        );

        return $this;
    }

    /**
     * Generate the stubbed file, save, and run migration.
     *
     * @return boolean
     */
    public function run()
    {
        $stub = file_get_contents(__DIR__ . "/stubs/{$this->mode}.txt");
        $name = strtolower("custom_fields_{$this->mode}_{$this->group->table_name}_table");

        $this->sections['table_name'] = $this->group->table_name;
        $this->sections['migration_name'] = studly_case($name);

        // Don't run a migration if we have nothing to do.
        if ($this->mode != 'drop' && empty($this->sections['up'])) {
            return false;
        }

        // Go through each of the template sections
        foreach ($this->sections as $key => $section) {
            // If an array, transform to string with appropriate enough spacing.
            if (is_array($section)) {
                $section = implode("\n        ", $section);
            }
            // Replace the template with the new data.
            $stub = str_replace('$' . mb_strtoupper($key) . '$', $section, $stub);
        }

        // Generate the file
        $filename = date('Y_m_d_His') . '_' . $name . '.php';
        $filename = base_path() . '/database/migrations/' . $filename;
        file_put_contents($filename, $stub);

        // Run the migration
        \Artisan::call('migrate');

        return true;
    }

    /**
     * Delete/Drop a table.
     * Used when a group is completely removed.
     *
     * @return self
     */
    public function drop()
    {
        $this->mode = 'drop';
        $this->removeFields($this->group->fields->pluck('column_name'));
        return $this;
    }
}
