<?php
/**
 * Media Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core;

use Illuminate\Database\Eloquent\Model;
use Intervention\Image\ImageManager;
use Nicolaslopezj\Searchable\SearchableTrait;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Media Eloquent Model
 *
 * Central location to store information about uploaded media.
 *
 * @package  Mtc\Core
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Media extends Model
{

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'type',
        'src',
        'url_name',
        'parent_id',
        'parent_type',
    ];

    /**
     * Define image types we support for upload/display
     * This is used when 404 parses missing image requests
     * TODO: This also will restrict image uploads
     * TODO: SVG upload support
     *
     * @var string[] $supported_image_extensions
     */
    public static $supported_image_extensions = [
        'jpg',
        'jpeg',
        'png',
        'gif'
    ];

    /**
     * Get a list of all owning parent models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function parent()
    {
        return $this->morphTo();
    }

    /**
     * Get URL of the original size for the uploaded media file
     *
     * @return string
     */
    public function getUrlAttribute()
    {
        return asset('storage/', $this->src);
    }

    /**
     * Generate the URL for size so we don't have to show full size images everywhere
     * Fallback to full size if size or Media owner does not exist
     *
     * @param string $size_name Size name to fetch
     * @throws NotFoundHttpException Size does not exist
     * @return string url for the image
     */
    public function getSize($size_info)
    {
        // Find all sizes for this media through config
        $all_media_sizes = config('media.' . get_class($this->parent));

        // If the config for this class doesn't exist or the size is not defined
        if (!$all_media_sizes) {
            throw new NotFoundHttpException('Media sizes for this object does not exist', null, 404);
        }

        // if $size_info is an array of width & height
        if (is_array($size_info) && count($size_info) == 2) {
            // Find size that matches the width & height values
            $size = collect($all_media_sizes)
                ->where('width', $size_info[0])
                ->where('height', $size_info[1])
                ->first();


            if (!$size) {
                // Size is not found in whitelisted images. Send back this path as it will 404
                return $size_info . '/' . $this->url_name;
            }

            // return image path with image size in pixels
            $image_path = pathinfo($size['path'])['dirname'] . '/' . $size['width'] . 'x' . $size['height'];
            return $image_path . '/' . $this->url_name;
        }

        // filter selected size
        $size = collect($all_media_sizes)
            ->filter(function ($size) use ($size_info) {
                // We need to make sure that size is an array and its last part matches selected size
                return is_array($size) &&  pathinfo($size['path'])['basename'] === $size_info;
            })->first();

        if (!$size) {
            // Size is not found in whitelisted images. Send back this path as it will 404
            return $size_info . '/' . $this->url_name;
        }

        // Return the correct url of the image by replacing the original path with media path
        return $size['path'] . '/' . $this->url_name;

    }


    /**
     * Attempt to generate a size based on a given path.
     * Will throw 404 exception if sizes are not available
     * This uses fit approach which crops and re-sizes based on given dimensions
     *
     * @param string $size_path Size access path
     * @throws NotFoundHttpException Size does not exist
     * @throws \Exception method was not able to set up a path for saving file
     * @return mixed
     */
    public function generateImageSize($size_path)
    {
        $size = $this->getSizeFromPath($size_path);

        // Size was not found also by dimensions
        if (!$size) {
            throw new NotFoundHttpException('This media size does not exist', null, 404);
        }

        $manager = new ImageManager([
            'driver' => config('core.image_manager_driver')
        ]);

        // Make sure we can upload to the required path
        $destination_path = public_path("storage/$size_path/");
        if (!@mkdir($destination_path) && !is_dir($destination_path)) {
            throw new \Exception('Image destination path is not valid');
        }

        // Create image from original
        $image = $manager->make(public_path('storage/' . $this->src));

        // Resize to required size. Will be Square if height not provided
        $image->fit($size['width'], $size['height'] ?? null);

        // Save image to sizes path
        $image->save($destination_path  . $this->url_name);

        // Return image with relevant HTTP headers
        return $image->response();
    }

    /**
     * Crop image size based on given coordinates
     *
     * @param string $size_path Size which to crop
     * @param string[] $coordinates array off coordinates to crop
     * @return string cropped image url
     * @throws \Exception method was not able to set up a path for saving file
     */
    public function cropSize(string $size_path, array $coordinates)
    {
        /*
         * Coordinates are defined as x1, y1, x2, y2
         * We need to calculate the width & height for the crop area
         * Scaling down will be handled afterwards
         */
        $width = $coordinates[2] - $coordinates[0];
        $height = $coordinates[3] - $coordinates[1];

        $size = $this->getSizeFromPath($size_path);

        // Initialize image manager
        $manager = new ImageManager([
            'driver' => config('core.image_manager_driver')
        ]);

        $public_path = "/storage/$size_path/";

        // Make sure we can upload to the required path
        $destination_path = public_path($public_path);
        if (!@mkdir($destination_path) && !is_dir($destination_path)) {
            throw new \Exception('Image destination path is not valid');
        }

        // Create image from original
        $image = $manager->make(public_path('storage/' . $this->src));

        // Pass the width, height and top-left corner coordinates to the crop tool
        $image->crop($width, $height, $coordinates[0], $coordinates[1]);

        // Resize to required size. Will be Square if height not provided
        $image->resize($size['width'], $size['height'] ?? $size['width']);

        // Save image to sizes path
        $image->save($destination_path  . $this->url_name);

        // Return image url with timestamp for cache busting
        return $public_path . $this->url_name . '?' . time();

    }

    /**
     * Get the size array from given size path.
     * Function checks and returns given size config for this media object.
     *
     * @throws NotFoundHttpException Size was not found
     * @param string $size_path name of the size path
     * @return string[] size config array
     */
    private function getSizeFromPath($size_path)
    {

        // Find all sizes for this media through config
        $all_media_sizes = config('media.' . get_class($this->parent));

        // If the config for this class doesn't exist or the size is not defined
        if (!$all_media_sizes) {
            throw new NotFoundHttpException('This media size does not exist', null, 404);
        }

        // Convert to collection so we can filter and treat it as
        $size =  collect($all_media_sizes)
            ->where('path', $size_path)
            ->first();

        /* Size was not found by name, lets check if name was a dimension
         * Regex checks the requested size to match something/123x321
         * This way we can change the image size dynamically during development:
         * - change the image ratio in config
         * - reload image in a template that has dimensions passed instead of size name
         */
        if (!$size && preg_match('#[-a-zA-Z]+/([0-9]+)x([0-9]+)#', $size_path, $matches)) {
            $size = collect($all_media_sizes)
                ->where('width', (int)$matches[1])
                ->where('height', (int)$matches[2])
                ->first();
        }
        return $size;

    }
}
