<?php
/**
 * Seo Admin URL Controller
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\Seo;

use Illuminate\Support\Facades\Event;
use Illuminate\View\View;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Models\Seo\CustomUrl;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Control Urls for pages.
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class CustomUrlController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-urls'
     * permission.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-urls');
    }

    /**
     * List all nodes in system that have a url.
     * This way we can see what urls we can manage.
     *
     * @param Request $request incoming request
     * @return \Illuminate\Http\Response|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        Event::listen('core.builder.core.admin.seo.urls.head', function () {
            return 'core::admin.seo.urls.filter';
        });

        $url_query = Node::where('url', '!=', '')->orderBy('id', 'desc');

        if (!empty($request->input('url'))) {
            $url_query->where('url', 'like', '%' . $request->input('url') .'%');
        }

        return (new Builder('core.admin.seo.urls', $url_query, 'Custom Site Urls'))
            ->columns([
                    'url' => trans('fields.url'),
                    'nodeable_type' => trans('fields.type'),
                    'status' => trans('fields.status'),
                    'visibility' => trans('fields.visibility')
            ])
            ->data([
                'cannot_delete' => function ($node) {
                    return $node->nodeable_type !== CustomUrl::class;
                },
                'nodeable_type' => function ($node) {
                    return class_basename($node->nodeable_type);
                }
            ])
            ->view([
                'form_method' => 'GET',
            ]);
    }

    /**
     * Create a new custom url
     * This will allow creating a new custom url for specific types of controllers
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $custom_url_resources = array_flatten(Event::fire('core.seo.getCustomUrlControllers'), 1);
        $custom_url = new CustomUrl();
        $resource_name = 'core.admin.seo.urls';
        $action = route(
            "{$resource_name}.store",
            [$custom_url->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.urls.form',
                'name' => $resource_name,
                'item' => $custom_url,
                'node' => null,
                'resource_list' => $custom_url_resources,
                'taxonomies' => Taxonomy::all(),
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Save a custom url
     *
     * @param Request $request incoming request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $this->validate($request,[
            'type' => 'required',
            'taxonomies' => 'array',
            'taxonomies.*' => 'numeric',
            'url' => 'required|alpha_dash_url|unique:nodes,url'
        ]);

        $custom_url = CustomUrl::create([
            'public_controller' => $request->input('type')
        ]);

        $custom_url->node()->create([
            'url' => $request->input('url'),
            'custom_url' => 1
        ]);

        $custom_url->node
            ->taxonomies()
            ->sync($request->input('taxonomies'));

        $request->session()->flash('success', 'Url added');
        return redirect(route('core.admin.seo.urls.index'));
    }

    /**
     * Edit a url of a page.
     * This is done by editing node that has a url
     *
     * @param int $node_id node Id to edit
     * @return View
     */
    public function edit(int $node_id)
    {
        $node = Node::with('nodeable')->find($node_id);

        if ($node->nodeable_type == CustomUrl::class) {
            $node->nodeable_type = class_basename($node->nodeable->public_controller);
        }

        $resource_name = 'core.admin.seo.urls';
        $action = route(
            "{$resource_name}.update",
            [$node->id]
        );
        // A bit hacky approach of passing node as null and $node as item
        // This is done to remove node default fields whilst passing node to the view
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.urls.form',
                'name' => $resource_name,
                'node' => null,
                'item' => $node,
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Update the url of a node
     *
     * @param Request $request incoming request
     * @param int $node_id node id
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(Request $request, int $node_id)
    {
        $this->validate($request, [
            'url' => 'required|alpha_dash_url|unique:nodes,url,' . $node_id
        ]);

        $node = Node::find($node_id);
        $node->url = $request->input('url');
        $node->custom_url = 1;
        $node->save();

        $request->session()->flash('success', 'Url updated');
        return redirect(route('core.admin.seo.urls.index'));
    }

    /**
     * Delete a node.
     * This allows deleting a custom url
     *
     * @param Request $request incoming request
     * @param int $node_id node id
     * @return Response|ResponseFactory|RedirectResponse
     */
    public function destroy(Request $request, int $node_id)
    {
        $node = Node::findOrFail($node_id);
        $node->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Url has been deleted.");
        return redirect()->back();
    }
}
