<?php
/**
 * Controller for Users
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin;

use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Mtc\Core\Auth\User;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Http\Requests\Admin\StoreUser;

/**
 * Controller for Users in Admin
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class UserController extends Controller
{
    /**
     * Ensure this is only accessible to those with the permission to do so.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-users');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return (new Builder('core.admin.users', User::query()))
            ->setDeletePermission('delete-users')
            ->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param User $user Empty user model
     *
     * @return \Illuminate\View\View
     */
    public function create(User $user)
    {
        return (new ItemBuilder('core.admin.users', $user))
            ->view('core::admin.user.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreUser $request Validated Request
     * @param User      $user    Empty user model
     *
     * @return \Illuminate\Http\Response
     */
    public function store(StoreUser $request, User $user)
    {
        $user->fill($request->all())
            ->save();

        $user->roles()->sync($request->input('roles', []));

        return redirect(route('core.admin.users.index'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param User $user User Model
     *
     * @return \Illuminate\View\View
     */
    public function edit(User $user)
    {
        return (new ItemBuilder('core.admin.users', $user))
            ->view('core::admin.user.form');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreUser $request Validated Request
     * @param User      $user    User to update
     *
     * @return \Illuminate\View\View
     */
    public function update(StoreUser $request, User $user)
    {
        return $this->store($request, $user);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param User    $user    User to Delete
     * @param Request $request Incoming request
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(User $user, Request $request)
    {

        $success = false;
        if (stristr($user->email, '@mtcmedia.co.uk') === false) {
            $success = $user->delete();
        }

        if ($request->ajax() && $success === true) {
            return response('success', 200);
        } elseif ($request->ajax()) {
            return response('fail', 400);
        } elseif ($success === true) {
            $request->session()
                ->flash('success', "{$user->name} has been deleted.");
        } else {
            $request->session()
                ->flash('error', "{$user->name} has not been deleted.");
        }

        return redirect()->back();
    }
}
