<?php
/**
 * Class Currency
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

/**
 * Class Currency
 *
 * Currency model
 * Used to store currency rates and info in Database
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-30
 */
class Currency extends Model
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'currencies';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'code',
        'rate',
        'active',
        'locale'
    ];

    /**
     * Extend bootstraping of model
     */
    public static function boot()
    {
        parent::boot();

        // we need to clear cache for currency when it is changed
        static::saved(function (self $currency) {
            Cache::forget("currency_{$currency->code}");
        });
    }

    /**
     * Get the symbol for this currency
     *
     * @return string currency symbol
     */
    public function getSymbol()
    {
        $formatter = new \NumberFormatter($this->locale, \NumberFormatter::DECIMAL);
        return $formatter->getSymbol(\NumberFormatter::CURRENCY_SYMBOL);

    }

    /**
     * Get the currency object by code.
     * Cache currency to avoid unnecessary requests
     *
     * @param string $currency_code currency code to fetch
     * @return mixed Currency object or false if not found
     */
    public static function getCachedCurrencyByCode(string $currency_code)
    {
        return Cache::remember(
            "currency_{$currency_code}",
            60,
            function () use ($currency_code) {
                return self::where('code', $currency_code)->first();
            });

    }

    /**
     * Shorthand function for fetching the currency rate for currently selected currency
     *
     * @return float
     */
    public static function getCurrentRate()
    {
        $currency = self::getCachedCurrencyByCode(session('currency_selected'));
        return $currency->rate ?? 1;
    }
}
