<?php
/**
 * Web Routes
 *
 * This file is where you may define all of the routes that are handled
 * by your application. Just tell Laravel the URIs it should respond
 * to using a Closure or controller method. Build something great!
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

use Mtc\Core\Admin\Menu;
use Mtc\Core\Events\Admin\MenuLoading;
use Illuminate\Routing\Router;

/**
 * @var Router $router
 */
$router->group([
    'middleware' => ['web'],
], function (Router $router) {

    // Currency Change Route, e.g. /currency?rate=EUR
    $router->get('currency', CurrencyController::class . '@store');

    // Set a pattern wildcard for all storage file requests
    // This regex matches all alpha-dash files with full path (forward slash allowed)
    $router->pattern('storage_file', "[a-zA-Z0-9\/\.\-\_]+");

    // The storage path must match what is defined in config.filesystems.disks.public.root
    $router->get('storage/{storage_file}', MediaController::class . '@show');
    $router->resource('media', MediaController::class);

});
$router->group(
    [
        'prefix' => config('core.admin_prefix'),
        'middleware' => ['web', 'auth', 'permission:admin'],
        'as' => 'core.admin.'
    ],
    function (Router $router) {

        // Homepage
        $router->get(
            '/', function () {
                return view('core::admin.home');
            }
        )->name('home');

        $router->post('search', Admin\SearchController::class . '@search')->name('search');

        $router->resource('users', Admin\UserController::class);
        $router->resource('roles', Admin\RoleController::class);
        $router->resource('currencies', Admin\CurrencyController::class);

        $router->group(
            [
                'prefix' => 'custom-fields',
                'as' => 'custom-fields.',
                'middleware' => ['permission:manage-custom-fields'],
            ], function (Router $router) {
                $router->resource('groups', Admin\CustomFields\GroupController::class);
                $router->post('groups/tableExists', Admin\CustomFields\GroupController::class . '@postTableExists');
            }
        );

        $router->post('taxonomies/{taxonomy}', Admin\TaxonomyController::class . '@storeChild')
            ->name('taxonomy.storeChild');
        $router->resource('taxonomies', Admin\TaxonomyController::class);

        // SEO
        $router->group([
            'prefix' => 'seo',
            'as' => 'seo.',
            'middleware' => ['permission:manage-seo'],
        ], function (Router $router) {
            // SEO Defaults
            $router->resource('browse_content', Admin\Seo\BrowseContentController::class);
            $router->resource('defaults', Admin\Seo\DefaultsController::class);

            // Custom Url management
            $router->resource('urls', Admin\Seo\CustomUrlController::class);

            // Page meta value management (title, description, h1)
            $router->post('page_meta_tags/bulk_upload', Admin\Seo\PageController::class . '@bulkUpload')
                ->name('page_meta_tags.bulk_upload');
            $router->get('page_meta_tags/bulk_example', Admin\Seo\PageController::class . '@bulkExampleDownload')
                ->name('page_meta_tags.bulk_example');
            $router->resource('page_meta_tags', Admin\Seo\PageController::class);

            // Seo Settings
            $router->resource('settings', Admin\Seo\SettingsController::class);

            // Knowledge Graph
            $router->post('knowledge_graph/{knowledge_graph}', Admin\Seo\KnowledgeGraphController::class . '@storeChild')
                ->name('knowledge_graph.storeChild');
            $router->get('knowledge_graph/preview', Admin\Seo\KnowledgeGraphController::class . '@preview')
                ->name('knowledge_graph.preview');
            $router->resource('knowledge_graph', Admin\Seo\KnowledgeGraphController::class);
        });
    }
);

Event::listen(
    MenuLoading::class, function () {

        // Management section
        Menu::addSection(
            'Management',
            'management'
        );

        // General management - users, permissions, currencies, languages etc.
        Menu::addMenu(
            'management',
            'General',
            '',
            'general',
            'glyphicon glyphicon-wrench'
        );

        Menu::addSubMenu(
            'general',
            'Manage Users',
            'manage-users',
            route('core.admin.users.index'),
            'manage_users'
        );

        Menu::addSubMenu(
            'general',
            'Manage Roles',
            'manage-roles',
            route('core.admin.roles.index'),
            'manage_roles'
        );

        if (config('core.currencies_enabled', false)) {
            Menu::addSubMenu(
                'general',
                'Manage Currencies',
                'manage-currencies',
                action('\\Mtc\Core\Http\Controllers\Admin\CurrencyController@index'),
                'manage_currencies'
            );
        }


        Menu::addMenu(
            'management',
            'Custom Fields',
            'manage-custom-fields',
            'custom-fields',
            'glyphicon glyphicon-log-in'
        );
        Menu::addSubMenu(
            'custom-fields',
            'Groups',
            'manage-custom-field-groups',
            route('core.admin.custom-fields.groups.index'),
            'custom-field-groups'
        );
        Menu::addSubMenu(
            'custom-fields',
            'Taxonomies',
            'manage-custom-field-taxonomies',
            route('core.admin.taxonomies.index'),
            'custom-field-taxonomies'
        );

        // SEO menu items
        Menu::addMenu(
            'management',
            trans('Seo'),
            'manage-seo',
            'seo-admin',
            'glyphicon glyphicon-eye-open'
        );
        Menu::addSubMenu(
            'seo-admin',
            trans('Seo Settings'),
            'manage-seo',
            route('core.admin.seo.settings.index'),
            'manage-seo-settings'
        );
        Menu::addSubMenu(
            'seo-admin',
            trans('Custom URLs'),
            'manage-seo',
            route('core.admin.seo.urls.index'),
            'manage-urls'
        );
        Menu::addSubMenu(
            'seo-admin',
            trans('Listing Page Content'),
            'manage-seo',
            route('core.admin.seo.browse_content.index'),
            'manage-seo-browse-content'
        );
        Menu::addSubMenu(
            'seo-admin',
            trans('Seo Defaults'),
            'manage-seo-defaults',
            route('core.admin.seo.defaults.index'),
            'manage-seo-defaults'
        );
        Menu::addSubMenu(
            'seo-admin',
            trans('Page Metadata'),
            'manage-seo',
            route('core.admin.seo.page_meta_tags.index'),
            'manage-seo-pages'
        );
        Menu::addSubMenu(
            'seo-admin',
            trans('Knowledge Graph'),
            'manage-seo',
            route('core.admin.seo.knowledge_graph.index'),
            'manage-seo'
        );

    }
);
