<?php
/**
 * Seo Admin Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\Seo;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Http\Requests\Admin\StoreSeoDefault;
use Mtc\Core\Models\Seo\Defaults;
use Mtc\Core\Node;

/**
 * Control Seo defaults used on site.
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class DefaultsController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-seo'
     * permission.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-seo');
    }

    /**
     * Render a browse page showing all the defaults available in the system.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return (new Builder('core.admin.seo.defaults', Defaults::query(), 'Seo Defaults'))
            ->columns(
                [
                'id' => trans('fields.id'),
                'name' => trans('fields.name'),
                'node_type' => trans('fields.type'),
                'path' => trans('fields.path'),
                'title' => trans('fields.title'),
                'description' => trans('fields.description'),
                ]
            )
            ->view();
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $default = new Defaults();
        $resource_name = 'core.admin.seo.defaults';
        $action = call_user_func_array(
            'route', [
                "{$resource_name}.store",
                [$default->id]
            ]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.defaults.form',
                'name' => $resource_name,
                'item' => $default,
                'node' => null,
                'node_list' => Node::distinct('nodeable_type')->pluck('nodeable_type'),
                'supported_tags' => Defaults::getSupportedReplaceTags(),
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Create a new Role
     *
     * @param StoreSeoDefault $request Request with validation
     * @param Defaults $default Empty seo default model
     *
     * @return \Illuminate\Http\Response Redirect to seo default listing
     */
    public function store(StoreSeoDefault $request, Defaults $default)
    {
        $default->fill($request->all())
            ->save();

        $request->session()->flash('success', 'Defaults updated for ' . $default->name);
        return redirect(route('core.admin.seo.defaults.index'));
    }

    /**
     * Show a view to allow the admin to edit an existing seo default instance.
     *
     * @param Defaults $default seo default instance to edit
     *
     * @return \Illuminate\View\View
     */
    public function edit(Defaults $default)
    {
        $resource_name = 'core.admin.seo.defaults';
        $action = call_user_func_array(
            'route', [
                "{$resource_name}.update",
                [$default->id]
            ]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.defaults.form',
                'name' => $resource_name,
                'item' => $default,
                'node' => null,
                'node_list' => Node::distinct('nodeable_type')->pluck('nodeable_type'),
                'supported_tags' => Defaults::getSupportedReplaceTags(),
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Save an existing seo default instance.
     *
     * @param StoreSeoDefault $request Validated request
     * @param Defaults $defaults Existing seo default instance
     *
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function update(StoreSeoDefault $request, Defaults $default)
    {
        return $this->store($request, $default);
    }

    /**
     * Delete an existing seo defaults instance
     *
     * @param Request $request Normal request
     * @param Defaults $default Seo Defaults instance
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Defaults $default)
    {
        $default->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Default settings for '{$default->name}' has been deleted.");
        return redirect()->back();
    }
}
