<?php
/**
 * Class SeoComposer
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Http\ViewComposers;

use Illuminate\Support\Facades\Cache;
use Illuminate\View\View;
use Mtc\Core\Models\Seo\Setting;
use Mtc\Core\Node;
use Mtc\Core\Models\Seo\Defaults;

/**
 * Class SeoComposer
 *
 * Add in the Seo information for views
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class SeoComposer
{
    /**
     * Handle the view processing to add in required Seo fields
     *
     * @param View $view
     */
    public function compose(View $view)
    {
        $view->with([
            'seo_data' => self::getSeoData(),
            'seo_tags' => self::setSeoTags()
        ]);
    }

    /**
     * Generate Seo Data for current page
     *
     * @return object
     */
    public static function getSeoData()
    {
        // Get the URL of the page
        $url = request()->getPathInfo();

        // Check if node exists for this url
        $node = Node::where('url', $url)
            ->first();

        // Check Fallback to Seo Defaults
        $defaults = $node ? Defaults::getNodeDefaults($node) : Defaults::getDefaultsFromUrl($url);
        if ($defaults) {
            return $defaults;
        }

        // Final fallback to Site Config
        return (object)[
            'title' => config('app.name'),
            'description' => config('app.name')
        ];
    }

    /**
     * Register all necessary 3rd party scripts that need to be included on site.
     * This if meant for functionalities like google tag manager.
     * The result of this function is an array of head and body sections (where script should be included)
     * Each section may consist of an array which has template and "with" values.
     *
     * @return array of tags to include for seo/marketing purposes
     */
    public static function setSeoTags()
    {
        $seo_tags = [
            'head' => [], // Scripts that need to be in head
            'body' => []  // Scripts that need to be in body
        ];

        /*
         * Check if Google Tag Manager has been enabled.
         * If it is enabled, pass the tag manager for inclusion in template
         * Cache this value for 1 hour. If it is changed from admin it will be cleared
         */
        $google_tag_manager = Cache::remember('GTM_CONTAINER_ID', 60, function () {
            return Setting::where('key', 'GTM_CONTAINER_ID')
                ->where('value', '!=', '')
                ->first();
        });

        if ($google_tag_manager) {
            $seo_tags['body'][] = [
                'template' => 'core::public.seo.google_tag_manager',
                'with' => [
                    'GTM_CONTAINER_ID' => $google_tag_manager->value
                ]
            ];
        }

        // Return all enabled seo tags
        return $seo_tags;
    }
}
