<?php
/**
 * Class Seo\Defaults
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Models\Seo;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;

/**
 * Class Seo\Defaults
 *
 * Seo Defaults model
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Defaults extends Model
{

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'seo_defaults';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'node_type',
        'path',
        'title',
        'description'

    ];

    /**
     * Get the seo defaults that are set for this node
     *
     * @param Node $node
     * @return Defaults|null
     */
    public static function getNodeDefaults(Node $node)
    {
        $default = self::where('node_type', $node->nodeable_type)
            ->first();

        if ($default) {
            $default->convertTagsToValues($node);
        }
        return $default;
    }

    /**
     * Get the Seo Defaults that are set for this url
     *
     * Uses inverse matching (/browse/page-2 must match the entry for /browse)
     *
     * @param $url
     * @return mixed
     */
    public static function getDefaultsFromUrl($url)
    {
        $url = DB::getPdo()->quote($url);
        $default = self::where('path', '!=', '')
            ->whereRaw(DB::raw($url . ' LIKE CONCAT(`path`, "%")'))
            ->first();

        return $default;
    }

    /**
     * Fill the default values with real values from node
     *
     * @param Node|false $node Node for current page
     * @param string $url url of this page
     */
    protected function convertTagsToValues($node, $url = '')
    {
        // Get supported Tags &
        $from = self::getSupportedReplaceTags()->toArray();
        ksort($from);

        // Start setting values for filling out placeholders
        $to = [
            'title' => $node->title,
            'description' => $node->description,
            'site_name' => config('app.name'),
        ];

        // Loop through all $from placeholders and replace ones that are not available with emptiness
        foreach ($from as $key => $value) {
            if (!isset($to[$key])) {
                $to[$key] = '';
            }
        }
        ksort($to);

        // convert tags to values
        $this->title = str_replace($from, $to, $this->title);
        $this->description = str_replace($from, $to, $this->description);

    }

    /**
     * Set up a list of all available SEO Tags
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getSupportedReplaceTags()
    {
        $tags = collect([
            'title' => '{TITLE}',
            'description' => '{DESCRIPTION}',
            'site_name' => "{SITE_NAME}",
            'page' => "{PAGE}"
        ]);

        // Loop through Taxonomies
        Taxonomy::roots()
            ->get()
            ->each(function ($taxonomy) use ($tags) {
                $tags->put('txr_' . $taxonomy->id, '{TX:[' . $taxonomy->title . ']:ROOT}');
                $tags->put('txc_' . $taxonomy->id, '{TX:[' . $taxonomy->title . ']:CURRENT}');
            });

        return $tags;
    }
}
