<?php
/**
 * Admin Menu
 *
 * Use this to generate the Admin Menu within the sidebar.
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Admin;

use Auth;
use Mtc\Core\Events\Admin\MenuLoading;

/**
 * Used to generate Admin menus.
 *
 * @category Mtc\Core\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Menu
{
    /**
     * Menu sections/groups (e.g. Shop / MTC Only)
     *
     * @var array
     */
    private static $sections = [];
    /**
     * Stores top-level menu items/links
     *
     * @var array
     */
    private static $items = [];
    /**
     * Stores child-level menu items/links
     *
     * @var array
     */
    private static $children = [];

    /**
     * Display the Menu on the front-end, triggered within the Blade template.
     *
     * @return \Illuminate\View\View
     */
    public static function render()
    {
        // Trigger an event to get each component to register their menus.
        event(new MenuLoading);

        // Get a list of unique section slugs
        $slugs = collect(self::$items)
            ->pluck('section_slug')
            ->toArray();

        // Get sections that only has menu items
        $sections = collect(self::$sections)
            ->filter(function($section) use ($slugs) {
                return in_array($section['section_slug'], $slugs);
            });

        return view('core::layouts.partials.admin.sidebar')
            ->with(
                [
                    'sections' => $sections,
                    'items' => collect(self::$items),
                    'children' => self::$children,
                ]
            );
    }

    /**
     * Add a new section to the menu
     *
     * @param string $section_title A title for the section
     * @param string $section_slug  A unique slug/id for reference when adding
     *                              menu items later
     *
     * @return void
     */
    public static function addSection($section_title, $section_slug)
    {
        self::$sections[] = compact('section_title', 'section_slug');
    }

    /**
     * Add a new top-level menu item
     *
     * @param string $section_slug The slug name for the parent section.
     * @param string $menu_title   The text to be used for the menu.
     * @param string $capability   The permission required for this menu to be
     *                             displayed to the user.
     * @param string $menu_slug    The slug name to refer to this menu by
     *                             (should be unique for this menu).
     * @param string $icon_class   The class to apply to this menu link.
     * @param int    $position     The position in the menu order this one
     *                             should appear.
     *
     * @return boolean             False if unable due to user permissions.
     */
    public static function addMenu(
        $section_slug,
        $menu_title,
        $capability,
        $menu_slug,
        $icon_class = '',
        $position = null
    ) {

        // Check if the user can do this action.
        if (!empty($capability) && false === Auth::user()->can($capability)) {
            return false;
        }

        // Define a default icon if one has not been provided.
        $icon_class = $icon_class ?: 'glyphicon-th-list';

        $new_menu = [
            'menu_title' => $menu_title,
            'capability' => $capability,
            'section_slug' => $section_slug,
            'menu_slug' => $menu_slug,
            'class' => $icon_class,
            'active' => false,
        ];

        // Determine the item's location in the array or add to the end.
        if ($position === null) {
            self::$items[] = $new_menu;
        } elseif (isset(self::$items["$position"])) {
            // Generate a random ID to ensure a unique position without overwrite
            $id = substr(base_convert(md5($menu_slug . $menu_title), 16, 10), -5);
            $position = $position + $id * 0.00001;
            self::$items["$position"] = $new_menu;
        } else {
            self::$items[$position] = $new_menu;
        }
    }

    /**
     * Add a child-level menu item.
     *
     * @param string $parent_slug The slug name for the parent menu.
     * @param string $menu_title  The text to be used for the menu.
     * @param string $capability  The permission required for this menu to be
     *                             displayed to the user.
     * @param string $menu_url    The URL to the requested resource
     * @param string $menu_slug   The slug name to refer to this menu by
     *                             (should be unique for this menu).
     *
     * @return boolean             False if unable due to user permissions.
     */
    public static function addSubMenu(
        $parent_slug,
        $menu_title,
        $capability,
        $menu_url,
        $menu_slug
    ) {
        // Check if the user can do this action.
        if (!empty($capability) && false === Auth::user()->can($capability)) {
            return false;
        }

        $active = $menu_url == url()->current();

        self::$children[$parent_slug][] = [
            'menu_title' => $menu_title,
            'capability' => $capability,
            'menu_slug' => $menu_slug,
            'menu_url' => $menu_url,
            'active' => $active,
        ];

        // If active, let the main item know.
        if ($active === true) {
            foreach (self::$items as &$item) {
                if ($item['menu_slug'] == $parent_slug) {
                    $item['active'] = true;
                    break;
                }
            }
        }

        return true;
    }
}
