<?php
/**
 * Core Service Provider
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Providers
 * @package  Mtc\Core\Admin
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Providers;

use App;
use Cache;
use Blade;
use Event;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Mtc\Core\Console\Commands\TriggerNpmIncludeAndInstall;
use Mtc\Core\Console\Commands\UpdateCurrencies;
use Mtc\Core\Models\Currency;
use Mtc\Core\Models\Seo\Defaults;
use Mtc\Core\Models\Seo\PageMetaValues;
use Mtc\Core\Models\Seo\Setting;
use Mtc\Core\Node;
use Mtc\Core\Core;
use Mtc\Core\Http\ViewComposers\RolesPermissionsComposer;
use Mtc\Core\Listeners\DeleteCustomData;
use Mtc\Core\Listeners\DeleteNode;
use Mtc\Core\Listeners\SaveCustomData;
use Mtc\Core\Listeners\ValidateMtcLogin;
use View;

/**
 * Connect Core to Laravel, registering routes and services that require access.
 *
 * @category Mtc\Core\Providers
 * @package  Mtc\Core\Admin
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class CoreServiceProvider extends ServiceProvider
{
    /**
     * Boot the service provider.
     *
     * @return void
     */
    public function boot()
    {
        $this->defineRoutes();
        $this->defineResources();
        $this->definePublished();
        $this->defineCommands();

        // Define a custom 'transpose' macro for Collections.
        Collection::macro(
            'transpose', function () {
                $items = array_map(
                    function (...$items) {
                        return $items;
                    }, ...$this->values()
                );

                return new static($items);
            }
        );

        // Set the currency output
        Blade::directive('currency', function ($value) {
            return "{{ money_format('%.2n', e(\Mtc\Core\Models\Currency::getCurrentRate() * $value) / 100) }}";
        });

    }

    /**
     * Define the command line actions that are set up in this package
     *
     * @return void
     */
    protected function defineCommands()
    {
        $this->commands([
            TriggerNpmIncludeAndInstall::class,
            UpdateCurrencies::class
        ]);
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        if (! defined('CORE_PATH')) {
            define('CORE_PATH', realpath(__DIR__.'/../../'));
        }

        if (! class_exists('Core')) {
            class_alias(Core::class, 'Core');
        }

        if ($this->app->runningInConsole()) {
            $this->commands([]);
        }

        // Register the Laravel debug bar when not production
        if ($this->app->environment('production') === false) {
            $this->app->register('Barryvdh\Debugbar\ServiceProvider');
        }


    }

    /**
     * Define the routes.
     *
     * @return void
     */
    protected function defineRoutes()
    {
        $this->defineRouteBindings();

        /**
         * If the routes have not been cached, we will include them in a route group
         * so that all of the routes will be conveniently registered to the given
         * controller namespace. After that we will load the Core routes file.
         */
        if (! $this->app->routesAreCached()) {
            Route::group(
                ['namespace' => 'Mtc\Core\Http\Controllers'],
                function ($router) {
                    $path = realpath(CORE_PATH . '/routes');
                    foreach (glob("{$path}/*.php") as $file) {
                        include $file;
                    }
                }
            );
        }
    }

    /**
     * Define the route model bindings.
     *
     * @return void
     */
    protected function defineRouteBindings()
    {
        Route::model('page_meta_tag', PageMetaValues::class);
        Route::model('default', Defaults::class);
        Route::model('setting', Setting::class);
    }

    /**
     * Define the resources for this package.
     *
     * @return void
     */
    protected function defineResources()
    {
        $this->loadViewsFrom(CORE_PATH . '/resources/views', 'core');
        $this->loadTranslationsFrom(CORE_PATH . '/resources/lang', 'core');
    }


    protected function definePublished()
    {
        // Set timestamp 1 second in future to ensure other vendors (like spatie permissions) go first
        $timestamp = date('Y_m_d_His', time() + 1);

        $this->publishes([
            __DIR__ . '/../../config/core.php' => config_path('core.php')
        ], 'config');


        $this->publishes([
            __DIR__ . '/../../database/migrations/' => database_path('migrations'),
        ], 'migrations');

        if (!class_exists('AddPermissionGroupsTable')) {
            $this->publishes([
                dirname(dirname(__DIR__)) . '/database/migrations/stubs/add_permission_groups_table.php.stub' => $this->app->databasePath() . "/migrations/{$timestamp}_add_permission_groups_table.php",
            ], 'migrations');
        }

        // Make sure we are publishing database seeds
        $this->publishes([
                __DIR__ . '/../../database/seeds/' => database_path('seeds')
            ], 'seeds'
        );
    }
}
