<?php
/**
 * Address Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers;

use Auth;
use Illuminate\Http\Request;
use Mtc\Core\Auth\User;
use Mtc\Core\Country;
use Mtc\Core\Models\User\Address;

/**
 * Class Address Controller
 *
 * Manages the User Addresses
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 */
class AddressController extends Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->breadcrumbs->addCrumb(ucwords(trans('core::text.addresses')), action('\\' . self::class . '@index'));
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $addresses = Auth::user()->addresses;
        $countries = Country::getOrderedActiveCountries();
        return view('core::public.addresses.index')->with(compact('addresses', 'countries'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->breadcrumbs->addCrumb(trans('core::text.create_new_address'), '');
        $countries = Country::getOrderedActiveCountries();
        $page_details = [
            'route' => route('core.member.addresses.store'),
            'method' => 'POST'
        ];

        $address = new Address();

        $is_default = false;
        if (Auth::user()->default_address) {
            $is_default = (Auth::user()->default_address == $address->id) ? true : false;
        }

        return view('core::public.addresses.form')->with(compact('page_details', 'address', 'countries', 'is_default'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param Address $address
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Address $address)
    {
        return $this->update($request, $address);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param Address $address
     * @return \Illuminate\Http\Response
     * @internal param int $id
     */
    public function update(Request $request, Address $address)
    {
        $country = $request->input('addresses.country');
        $this->validate(
            $request,
            Address::getAddressValidationRules($country, ''),
            [],
            Address::getAddressValidationFieldNames('')
        );

        $address->fill($request->input('addresses'));
        $address->user_id = Auth::user()->id;
        $address->save();

        return redirect(route('core.member.addresses.index'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Address $address
     * @return \Illuminate\Http\Response
     */
    public function edit(Address $address)
    {
        if ($address->user->id == Auth::user()->id) {
            $this->breadcrumbs->addCrumb(trans('core::text.edit_address'), '');
            $countries = Country::getOrderedActiveCountries();
            $page_details = [
                'route' => route('core.member.addresses.update', $address->id),
                'method' => 'PUT'
            ];

            return view('core::public.addresses.form')->with(compact('page_details', 'address', 'countries',
                'is_default'));
        }
        return redirect(route('core.member.addresses.index'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param Address $address
     * @return \Illuminate\Http\Response
     * @internal param int $id
     */
    public function destroy(Request $request, Address $address)
    {
        if ($address->user->id == Auth::user()->id) {
            // delete address
            $address->delete();
            // is the address was set as default
            if ($address->id == Auth::user()->default_address) {
                // set default_address to 0 i.e. false
                User::where('id', Auth::user()->id)->update(['default_address' => 0]);
            }
            if ($request->ajax()) {
                return response('success', 200);
            }
            $request->session()->flash('success', trans('core::text.address_delete_success'));
        }
        return redirect(route('core.member.addresses.index'));
    }

    /**
     * Sets the current users default address
     *
     * @param Request $request
     * @param Address $address
     */
    public function setDefault(Request $request)
    {
        $address = Address::where('id', $request->input('id'))->first();
        if ($address->user->id == Auth::user()->id) {
            User::where('id', Auth::user()->id)->update(['default_address' => $address->id]);
        }
        return redirect(route('core.member.addresses.index'));
    }
}
