<?php
/**
 * Browse Content Controller
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\Seo;

use Illuminate\Support\Collection;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Models\Seo\BrowseContent;
use Mtc\Core\Taxonomy;

/**
 * Control Browse page content.
 *
 * This controller allows managing content on browse pages
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class BrowseContentController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-seo'
     * permission.
     */
    public function __construct()
    {
        $this->middleware('permission:manage-seo');
    }

    /**
     * Render a browse page showing all the content configurations available in the system.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return (new Builder('core.admin.seo.browse_content', BrowseContent::query(), 'Listing Page Content'))
            ->columns([
                'taxonomies' => trans('fields.taxonomies'),
                'title' => trans('fields.title'),
                'description' => trans('fields.description'),
            ])
            ->data([
                'taxonomies' => function ($item) {
                    /**
                     * @var Collection $taxonomies
                     */
                    $taxonomies = $item->taxonomies->map(function ($taxonomy) {
                        return $taxonomy->title;
                    });
                    return implode(', ', $taxonomies->toArray());
                },
                'title' => function ($item) {
                    return $item->title;
                },
                'description' => function ($item) {
                    return str_limit($item->description);
                },

            ])
            ->view();
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $content = new BrowseContent();
        $resource_name = 'core.admin.seo.browse_content';
        $action = route(
            "{$resource_name}.store",
            [$content->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.browse_content.form',
                'name' => $resource_name,
                'item' => $content,
                'node' => null,
                'taxonomies' => Taxonomy::all(),
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Create a new Role
     *
     * @param Request $request Request with validation
     * @param BrowseContent $browse_content Empty seo default model
     * @throws \Exception Unable to save session
     * @return \Illuminate\Http\Response Redirect to seo default listing
     */
    public function store(Request $request, BrowseContent $browse_content)
    {
        $this->validate($request, [
            'description' => 'required',
            'taxonomies' => 'required|array'
        ]);

        $browse_content->fill($request->all())
            ->save();

        $browse_content->taxonomies()->sync($request->input('taxonomies'));

        $request->session()->flash('success', 'Content updated');
        return redirect(route('core.admin.seo.browse_content.index'));
    }

    /**
     * Show a view to allow the admin to edit an existing seo default instance.
     *
     * @param BrowseContent $browse_content Content instance to edit
     *
     * @return \Illuminate\View\View
     */
    public function edit(BrowseContent $browse_content)
    {
        $resource_name = 'core.admin.seo.browse_content';
        $action = route(
            "{$resource_name}.update",
            [$browse_content->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.browse_content.form',
                'name' => $resource_name,
                'item' => $browse_content,
                'node' => null,
                'taxonomies' => Taxonomy::all(),
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Delete an existing seo browse content instance
     *
     * @param Request $request Normal request
     * @param BrowseContent $browse_content Seo Browse Content instance
     * @throws \Exception unable to delete
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, BrowseContent $browse_content)
    {
        $browse_content->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', 'Content entry has been deleted.');
        return redirect()->back();
    }

    /**
     * Save an existing seo browse Content instance.
     *
     * @param Request $request Incoming Request
     * @param BrowseContent $browse_content Existing content instance
     * @throws \Exception Unable to complete storing request
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function update(Request $request, BrowseContent $browse_content)
    {
        return $this->store($request, $browse_content);
    }
}
