<?php
/**
 * Knowledge Graph Admin Controller
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\Seo;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Models\Seo\KnowledgeGraph;

/**
 * Knowledge Graph Admin Controller
 *
 * Manage Knowledge Graph on site
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class KnowledgeGraphController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-seo'
     * permission.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-seo');
    }

    /**
     * Render a browse page showing all the defaults available in the system.
     *
     * @return \Illuminate\Http\Response|array array for ajax request
     */
    public function index(Request $request)
    {
        if ($request->wantsJson()) {
            $data = KnowledgeGraph::roots()->get();

            return [
                'children' => $data->map(
                    function ($item) {
                        $item->isExpanded = false;
                        return $item;
                    }
                )
            ];
        }
        if ($request->input('generate')) {
            KnowledgeGraph::generateGraph();
        }

        return view('core::admin.seo.knowledge_graph.index', [
            'knowledge_graph_preview' => json_encode($this->preview(), JSON_PRETTY_PRINT),
            'saved_knowledge_graph' => KnowledgeGraph::currentPublicGraph()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        KnowledgeGraph::create(
            [
                'field' => $request->input('field'),
                'type' => $request->input('type'),
            ]
        );

        return $this->index($request);
    }

    /**
     * Display the specified resource.
     *
     * @param Request  $request  Illuminate Request
     * @param KnowledgeGraph $knowledge_graph KnowledgeGraph Model
     *
     * @return \Illuminate\Http\Response JSON data
     */
    public function show(Request $request, KnowledgeGraph $knowledge_graph)
    {
        $knowledge_graph->children->map(
            function ($child) {
                // Set a default value for Vue to hook into.
                $child->isExpanded = false;
                return $child;
            }
        );
        return $knowledge_graph;
    }

    /**
     * Remove the specified resource from storage.
     *
     * If AJAX request, show a success message.
     *
     * @param Request  $request  Illuminate Request
     * @param KnowledgeGraph $knowledge_graph Knowledge Graph Model
     *
     * @return \Illuminate\Http\Response|void
     */
    public function destroy(Request $request, KnowledgeGraph $knowledge_graph)
    {
        $knowledge_graph->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, KnowledgeGraph $knowledge_graph)
    {
        // make sure we have a parent element
        $parent_type = $knowledge_graph->parent->type ?? '';

        if ($parent_type === 'list' && !empty($request->input('field'))) {
            $knowledge_graph->field = $request->input('field');
        } else {
            $knowledge_graph->value = $request->input('value');
        }
        $knowledge_graph->save();

        // If we have a parent element we're changing a child and we can use show()
        // If we have a top level element we need to use index() as it finds root elements
        if ($knowledge_graph->parent) {
            return $this->show($request, $knowledge_graph->parent);
        } else {
            return $this->index($request, $knowledge_graph->parent);

        }
    }

    /**
     * Create a child category.
     *
     * @param Request  $request  Request
     * @param KnowledgeGraph $knowledge_graph Knowledge Graph Model
     *
     * @return \Illuminate\Http\Response JSON data
     */
    public function storeChild(Request $request, KnowledgeGraph $knowledge_graph)
    {
        if ($knowledge_graph->type === 'list') {

        }
        $knowledge_graph->children()->create(
            [
                'field' => $request->input('field'),
                'type' => $request->input('type')
            ]
        );
        return $this->show($request, $knowledge_graph);
    }

    /**
     * Create a child category.
     *
     * @param Request  $request  Request
     * @param Taxonomy $taxonomy Taxonomy Model
     *
     * @return \Illuminate\Http\Response JSON data
     */
    public function preview()
    {
        return KnowledgeGraph::renderTree();
    }
}
