<?php
/**
 * Validate MTC Login Listener
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Listeners
 * @package  Mtc\Core\Admin
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Listeners;

use Mtc\Core\Auth\User;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use Illuminate\Auth\Events\Attempting;

/**
 * During login determine if the user/pass is a match for details
 * within the MTC Projects System for company emails only. The
 * user will be automatically logged in even if no account.
 *
 * @category Mtc\Core\Listeners
 * @package  Mtc\Core\Admin
 * @author   Craig McCreath <craig.mccreath@mtcmeida.co.uk>
 */
class ValidateMtcLogin
{
    /**
     * The URL to send the raw credentials too to determine good login.
     *
     * @var string
     */
    protected $api_url = 'https://projects.mtcmedia.co.uk/api/staff_login.php';
    /**
     * The credentials provided by the Attempting event.
     *
     * @var array
     */
    protected $credentials = [];
    /**
     * The response from the API.
     *
     * @var null|array Will be updated once response is delivered
     *                 via checkLoginViaProjects
     */
    protected $response = null;
    /**
     * The user model.
     *
     * @var null|model Filled during findLocalUser()
     */
    protected $user = null;

    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     *
     * @param Attempting $event Triggered event
     *
     * @return void
     */
    public function handle(Attempting $event)
    {
        $this->credentials = $event->credentials;
        if (! $this->isMtc()) {
            return false;
        }

        $this->findLocalUser()
            ->checkLoginViaProjects()
            ->updateUser();
    }

    /**
     * Determine if the email provided is our own.
     *
     * @return bool
     */
    protected function isMtc()
    {
        return str_contains($this->credentials['email'], '@mtcmedia.co.uk')
            && !str_contains($this->credentials['email'], 'dev');
    }

    /**
     * Find if the user exists and if not, return a new model.
     *
     * @return self
     */
    protected function findLocalUser()
    {
        $class = config('auth.model');
        $this->user = $class::whereEmail($this->credentials['email'])->first()
                      ?: new $class;

        return $this;
    }

    /**
     * Run a call against the API and store the decoded JSON response.
     *
     * @return self
     */
    protected function checkLoginViaProjects()
    {
        try {
            $response = (new Client)->request(
                'POST',
                $this->api_url,
                ['form_params' => $this->credentials]
            );

            $this->response = json_decode($response->getBody(), true);
            if (! is_array($this->response)) {
                $this->response = null;
            }
        } catch (ClientException $e) {
            $this->response = false;
        } catch (ConnectException $e) {
            // As we cannot connect, we can't proceed with the response
            // Usually only happens if you're trying to work locally without
            // an internet connection!
            $this->response = false;
        }

        return $this;
    }

    /**
     * Update the user with the results of the response.
     *
     * @return void
     */
    protected function updateUser()
    {
        // No need to continue if user doesn't exist and bad response.
        if ($this->response === false
            || ($this->user->exists == false && is_null($this->response))
        ) {
            return;
        }

        // Update pass to a random string if we have a bad response.
        $this->user->password = is_null($this->response)
                                ? bcrypt(str_random(40))
                                : bcrypt($this->credentials['password']);

        // Add the rest of the params if the user doesn't exist already.
        if (! $this->user->exists) {
            $this->user->fill($this->response);
            $this->user->full_name = $this->response['name'];
        }

        $this->user->save();

        // Ensure mtc. users are assigned to the master role.
        if (false === $this->user->hasRole('mtc')) {
            $this->user->assignRole('mtc');
        }
    }
}
