<?php
/**
 * Address Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 */

namespace Mtc\Core\Traits;

/**
 * Parent class for storing information about an address.
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 */
trait AddressValidator
{
    /**
     * Validation rules for an address
     *
     * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
     * @author James Neill <james.neill@mtcmedia.co.uk>
     * @param string $country Country code
     * @param string $type Address type
     * @param string $prefix
     * @return array
     */
    public static function getAddressValidationRules($country, $type = 'billing', $prefix = 'address'): array
    {
        if (!empty($type)) {
            $prefix = $prefix . '.' . $type;
        }
        return [
            "$prefix.first_name" => 'required',
            "$prefix.last_name" => 'required',
            "$prefix.address1" => 'required',
            "$prefix.city" => 'required',
            "$prefix.postcode" => 'postcode:' . $country,
            "$prefix.country" => 'required',
        ];
    }

    /**
     * Validation rules field names for an address
     *
     * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
     * @param string $type Address type
     * @param string $prefix
     * @return array
     */
    public static function getAddressValidationFieldNames($type = 'billing', $prefix = 'address'): array
    {
        if (!empty($type)) {
            $prefix = $prefix . '.' . $type;
        }
        return [
            "$prefix.first_name" => trans('shop::fields.address.first_name'),
            "$prefix.last_name" => trans('shop::fields.address.last_name'),
            "$prefix.address1" => trans('shop::fields.address.address1'),
            "$prefix.city" => trans('shop::fields.address.city'),
            "$prefix.postcode" => trans('shop::fields.address.postcode'),
            "$prefix.country" => trans('shop::fields.address.country')
        ];
    }

    /**
     * Normalizes postcode format to uppercase with space (if supported by country).
     *
     * @param string $postcode_string postcode string
     * @return string normalized postcode
     */
    public static function normalizePostcode($postcode_string, $country)
    {
        switch ($country) {
            // United Kingdom
            // Brings dd13ja to DD1 3JA
            case "GB":
                // Make it upper case
                $postcode_string = strtoupper($postcode_string);

                // adjust postcode with no spaces
                if (strpos($postcode_string, ' ') === false) {
                    $postcode_string = preg_replace('#(.{3})$#', ' $1', $postcode_string);
                }
                break;
        }

        return $postcode_string;
    }

}
