<?php
/**
 * Web Routes
 *
 * This file is where you may define all of the routes that are handled
 * by your application. Just tell Laravel the URIs it should respond
 * to using a Closure or controller method. Build something great!
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

use Illuminate\Routing\Router;

/**
 * @var Router $router
 */
$router->group([
    'middleware' => ['web'],
], function (Router $router) {

    $router->get('/logout', Auth\LoginController::class . '@logout');
    $router->get('/destroy', HomeController::class . '@clearSession');

    // Currency Change Route, e.g. /currency?rate=EUR
    $router->get('currency', CurrencyController::class . '@store');

    // Set a pattern wildcard for all storage file requests
    // This regex matches all alpha-dash files with full path (forward slash allowed)
    $router->pattern('storage_file', "[a-zA-Z0-9\/\.\-\_]+");

    // Address Finder
    $router->get('find_address', AddressFinderController::class . '@index')->name('findAddress');
    $router->post('find_address', AddressFinderController::class . '@index')->name('findAddress');

    // List countries on site
    $router->get('list_countries', AddressFinderController::class . '@show')->name('listCountries');

});


$router->group([
    'prefix' => config('core.member.prefix'),
    'middleware' => ['web', 'auth'],
    'as' => 'core.member.'
], function (Router $router) {
    $router->get('/', Members\DashboardController::class . '@index')->name('dashboard');
    $router->resource('details', Members\DetailsController::class);

    // The storage path must match what is defined in config.filesystems.disks.public.root
    if (config('core.member.pages.addresses')) {
        $router->resource('/addresses', Members\AddressController::class, ['except' => ['index', 'show']]);
        $router->get('/list_addresses', Members\AddressController::class . '@show')->name('addresses.list');
        $router->post('/addresses/default', Members\AddressController::class . '@setDefault')->name('addresses.default');
    }
});

$router->group(
    [
        'prefix' => config('core.admin_prefix'),
        'middleware' => ['web', 'auth', 'permission:admin'],
        'as' => 'core.admin.'
    ],
    function (Router $router) {
        // Homepage
        $router->resource('/', Admin\HomeController::class);
        $router->get('show', Admin\HomeController::class . '@show')->name('stats');
        $router->get('chart', Admin\HomeController::class . '@chart')->name('chart');

        $router->get('search', Admin\SearchController::class . '@search')->name('search');

        $router->get('node_list', NodeController::class . '@list')->name('list_nodes');
        $router->get('node', NodeController::class . '@show')->name('show');

        $router->resource('users', Admin\UserController::class);
        $router->resource('roles', Admin\RoleController::class);
        $router->resource('emails', Admin\EmailController::class);
        $router->resource('currencies', Admin\CurrencyController::class);

        $router->group(
            [
                'prefix' => 'custom-fields',
                'as' => 'custom-fields.',
                'middleware' => ['permission:manage-custom-fields'],
            ], function (Router $router) {
            $router->resource('groups', Admin\CustomFields\GroupController::class);
            $router->post('groups/tableExists', Admin\CustomFields\GroupController::class . '@postTableExists');
        }
        );

        $router->post('taxonomies/process', Admin\TaxonomyController::class . '@process')->name('taxonomies.process');
        $router->post('taxonomies/change_parent', Admin\TaxonomyController::class . '@changeParent');
        $router->get('taxonomies/list', Admin\TaxonomyController::class . '@list')->name('taxonomies.list');
        $router->post('taxonomies/update_order', Admin\TaxonomyController::class . '@updateOrder')->name('taxonomies.reorder');
        $router->post('taxonomies/{taxonomy}', Admin\TaxonomyController::class . '@storeChild')
            ->name('taxonomy.storeChild');
        $router->resource('taxonomies', Admin\TaxonomyController::class);

        // SEO
        $router->group([
            'prefix' => 'seo',
            'as' => 'seo.',
            'middleware' => ['permission:manage-seo'],
        ], function (Router $router) {
            // SEO Defaults
            $router->resource('browse_content', Admin\Seo\BrowseContentController::class);
            $router->resource('defaults', Admin\Seo\DefaultsController::class);

            // Custom Url management
            $router->resource('urls', Admin\Seo\CustomUrlController::class);

            // Page meta value management (title, description, h1)
            $router->post('page_meta_tags/bulk_upload', Admin\Seo\PageController::class . '@bulkUpload')
                ->name('page_meta_tags.bulk_upload');
            $router->get('page_meta_tags/bulk_example', Admin\Seo\PageController::class . '@bulkExampleDownload')
                ->name('page_meta_tags.bulk_example');
            $router->resource('page_meta_tags', Admin\Seo\PageController::class);

            // Seo Settings
            $router->resource('settings', Admin\Seo\SettingsController::class);

            // Knowledge Graph
            $router->post('knowledge_graph/{knowledge_graph}', Admin\Seo\KnowledgeGraphController::class . '@storeChild')
                ->name('knowledge_graph.storeChild');
            $router->post('knowledge_graph/{knowledge_graph}/edit', Admin\Seo\KnowledgeGraphController::class . '@update')
                ->name('knowledge_graph.update');
            $router->get('knowledge_graph/preview', Admin\Seo\KnowledgeGraphController::class . '@preview')
                ->name('knowledge_graph.preview');
            $router->resource('knowledge_graph', Admin\Seo\KnowledgeGraphController::class);
        });
    }
);
