<?php
/**
 * CustomFieldValue Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\CustomFields;

use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Node;
use Mtc\Media\Media;
use Spatie\Translatable\HasTranslations;

/**
 * CustomFieldValue Eloquent Model
 *
 * Used to store custom field data.
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class CustomFieldValue extends Model
{
    use HasTranslations;

    /**
     * @var string[] $translatable fields that can be translated
     */
    public $translatable = [
        'value_translatable'
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        // Link to Node that has these fields
        'node_id',
        // CustomGroup parent
        'group_id',
        // CustomField parent
        'field_id',
        // Field Name - for search-ability
        'field_name',
        // It's type (text, textarea, wysiwyg, dropdown, etc)
        'value',
        'text_value',
    ];

    /**
     * @var string[] $morphed_relations
     */
    protected static $morphed_relations = [
        'image' => Media::class
    ];

    /**
     * The field's parent group.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    protected function group()
    {
        return $this->belongsTo(CustomGroup::class, 'group_id');
    }

    /**
     * Relationship to Field that describes the type of this field
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function field()
    {
        return $this->belongsTo(CustomField::class, 'field_id');
    }

    /**
     * Retrieve any related media.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function media()
    {
        return $this->morphMany(Media::class, 'parent')
            ->orderBy('order', 'asc');
    }

    /**
     * Relationship to the Node - object who has this field
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    protected function node()
    {
        return $this->belongsTo(Node::class, 'node_id');
    }

    public function scopeOfGroup($query, $group_id)
    {
        return $query->where('group_id', $group_id);
    }

    public function scopeOfNode($query, $node_id)
    {
        return $query->where('node_id', $node_id);
    }

    /**
     * Get the correct data from DB
     *
     * @return array|mixed
     */
    public function getDataValueAttribute()
    {

        if ($this->field->isText() || $this->field->translatable) {
            if ($this->field->translatable) {
                return $this->getTranslations('text_value');
            }

            return $this->text_value;
        }

        return $this->value;
    }

    /**
     * Add additional automatic changes when manipulating object
     */
    protected static function boot()
    {
        parent::boot();

        // Make sure we link the morphed object to this custom field when value is updated
        self::saved(function ($custom_field_value) {
            if (array_key_exists('value', $custom_field_value->getDirty()) && array_key_exists($custom_field_value->field->type, self::$morphed_relations)) {
                /** @var Model $model */
                $model = new self::$morphed_relations[$custom_field_value->field->type];
                $model = $model->find($custom_field_value->value);
                if ($model) {
                    $model->parent_id = $custom_field_value->id;
                    $model->save();
                }

                // Unlink old one here
                if ($custom_field_value->getOriginal('value')) {
                    $model = new self::$morphed_relations[$custom_field_value->field->type];
                    $model = $model->find($custom_field_value->getOriginal('value'));
                    if ($model) {
                        $model->parent_id = $custom_field_value->id;
                        $model->save();
                    }
                }

            }
        });

        // Remove the link to this field when it is deleted
        self::deleted(function ($custom_field_value) {
            if (array_key_exists($custom_field_value->field->type, self::$morphed_relations)) {
                /** @var Model $model */
                $model = new self::$morphed_relations[$custom_field_value->field->type];
                $model = $model->find($custom_field_value->value);
                if ($model) {
                    $model->parent_id = null;
                    $model->save();
                }
            }
        });
    }

}
