<?php
/**
 * Controller for CustomGroups
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\CustomFields;

use Cache;
use Mtc\Core\CustomFields\CustomFieldValue;
use Illuminate\View\View;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Database\Schema\Builder as SchemaBuilder;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Node;
use Mtc\Core\CustomFields\CustomField as Field;
use Mtc\Core\CustomFields\CustomGroup as Group;
use Mtc\Core\CustomFields\CustomGroupModel;
use Mtc\Core\Http\Requests\Admin\StoreCustomGroup;
use Schema;

/**
 * Deal with the generation of Custom Groups, and their tables.
 *
 * @category Mtc\Core\Http\Controllers\Admin\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class GroupController extends Controller
{
    /**
     * Ensure that this page is only accessible to those with the
     * 'manage-custom-field-groups' permission.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:manage-custom-field-groups');
    }

    /**
     * The route this is accessible from.
     *
     * @var string
     */
    protected static $route = 'core.admin.custom-fields.groups';

    /**
     * The resource's index page.
     *
     * Returns either the page, or list of models ordered by title in JSON.
     *
     * @param Request $request Illuminate Request
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $groups = Group::with('models')->orderBy('title');
            // If custom field knows for what type of node this is needed, filer those out
            if (!empty($request->input('group_model'))) {
                $groups->whereHas('models', function ($query) use ($request) {
                     $query->where('model', $request->input('group_model'));
                });
            }
            return $groups->get();
        }

        return (new Builder(self::$route, Group::query()))
            ->view();
    }

    /**
     * Show the 'Add Group' page
     *
     * @param Group $group Blank Group model
     *
     * @return \Illuminate\View\View
     */
    public function create(Group $group)
    {
        return (new ItemBuilder(self::$route, $group))
            ->view('core::admin.custom_fields.groups.form');
    }

    /**
     * Save a new group, generating a new migration and running in the process.
     *
     * @param StoreCustomGroup $request Incoming Request
     * @param Group            $group   Blank Group model
     *
     * @return \Illuminate\Http\Response
     */
    public function store(StoreCustomGroup $request, Group $group)
    {
        // Setup and save the item
        $group->fill($request->input('item', []));
        $group->save();

        // Delete any corresponding models and rebuild
        $group->models()->delete();
        collect($request->input('models'), [])
            ->each(
                function ($model) use ($group) {
                    $model = new CustomGroupModel(compact('model'));
                    $group->models()->save($model);
                }
            );

        // Go through each field and add to the migration
        $fields = $request->input('fields');
        collect(
            [
                $fields['ids'],
                $fields['title'],
                $fields['type'],
                $fields['list_id'],
                $fields['multiple'],
                $fields['translatable']
            ]
        )->transpose()
            ->reject(
                function ($field) {
                    // Remove any where title or column_name are empty
                    return empty($field[1]) || empty($field[2]);
                }
            )->each(
                // Save field to group and add to migration
                function ($item, $key) use ($group, &$fields) {
                    $field = Field::findOrNew($item[0]);
                    $field->title = $item[1];
                    $field->type = $item[2];
                    $field->order = $key;
                    $field->multiple = (bool)$item[4];
                    $field->translatable = (bool)$item[5];

                    $field->list_id = null;
                    if (!empty($item[3])) {
                        $field->list_id = $item[3];
                    }

                    $group->fields()->save($field);
                    $fields['ids'][] = $field->id;
                }
            );

            // Remove existing fields that were not part of the group anymore
            $group->fields()->whereNotIn('id', $fields['ids'])->delete();

            $request->session()->flash('info', 'Custom Field Group Saved');
            return redirect(route(self::$route . '.edit', $group));
    }

    /**
     * Return information about the group's fields and taxonomies.
     *
     * Taxonomies are cached for 2hrs, and cleared if taxonomies are updated
     * elsewhere.
     *
     * @param Request $request Incoming request object
     * @param Group   $group   CustomGroup, auto-filled by Laravel
     *
     * @return \Illuminate\Http\Response JSON Response
     */
    public function show(Request $request, Group $group)
    {
        foreach ($group->fields as $field) {
            if ($field->taxonomy) {
                /**
                 * Sadly, a cache is necessary here due to the crazy number of
                 * SQL queries that getDescendants() generates. This gets
                 * cleared if the taxonomy is changed.
                 */
                $field->taxonomy->items = Cache::remember(
                    "taxonomy:{$field->id}", 120, function () use ($field) {
                        return $field->taxonomy->getDescendants()->map(
                            function ($item) {
                                $padding = '-';
                                $item->depth--;

                                $item->title = str_repeat($padding, strlen($padding) * $item->depth);
                                $item->title .= ' ' . $item->node->title;

                                return [
                                    'id' => $item->id,
                                    'title' => $item->title,
                                ];
                            }
                        );
                    }
                );
            }
        }

        // Bring in the models that are lazy loaded.
        $group->models;

        $group_data = [];
        // Retrieve information about the node.
        if ($request->has('node_id') && ($node = Node::query()->find($request->input('node_id')))) {
            $node->group_id = $group->id;
            $group_data = $node->group_data;
        }

        /*
         * We need to loop through fields to check if all data entries are set
         * This way we ensure that all data entries are set when ajax sets the data in vue instance
         * And we are not encountering issues with saving data
         */
        foreach ($group->fields as $field) {
            if (empty($group_data[$field->id])) {
                $group_data[$field->id] = $field->multiple ? [] : '';
            }
        }

        $group->data = $group_data;

        // Allow Laravel to return JSON.
        return $group;
    }

    /**
     * Show the edit page.
     *
     * @param Group $group Custom Group
     *
     * @return \Illuminate\Http\Response|View
     */
    public function edit(Group $group)
    {
        $group->fields;
        return $this->create($group);
    }

    /**
     * Update the existing model.
     *
     * @param StoreCustomGroup $request Incoming request
     * @param Group            $group   Custom group
     *
     * @return \Illuminate\Http\Response
     */
    public function update(StoreCustomGroup $request, Group $group)
    {
        return $this->store($request, $group);
    }

    /**
     * Delete the current group, and in turn create a new migration to drop
     * the table which will then run.
     *
     * @param Request $request Incoming Request
     * @param Group   $group   Custom group
     *
     * @return \Illuminate\Http\Response
     * @throws \Exception
     */
    public function destroy(Request $request, Group $group)
    {
        $group->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "'The item has been deleted.");
        return redirect()->back();
    }

    /**
     * Check if a table by the provided name exists in the current db. Used to
     * show success/fail message to Vue.
     *
     * @param Request $request Incoming Request
     *
     * @return \Illuminate\Http\Response
     */
    public function postTableExists(Request $request)
    {
        return response()->json(
            Schema::hasTable($request->input('table', ''))
        );
    }
}
