<?php
/**
 * Admin dashboard controller
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk
 */
namespace Mtc\Core\Http\Controllers\Admin;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Event;
use Mtc\Core\Events\Admin\FetchDashboardStats;
use Mtc\Core\Http\Controllers\Controller;

/**
 * Admin dashboard controller
 *
 * Admin landing page controller
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk
 * @version 2017-06-17
 */
class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {

        return view('core::admin.home', [
            'dashboard_stats' => [],
            'active_filter' => session('admin.dashboard.filter') ?? 'week',
        ]);
    }

    /**
     * Fetch the stats data
     * Store the selected date range in session
     *
     * @param Request $request
     * @return array
     */
    public function show(Request $request)
    {
        $end = Carbon::now();
        switch ($request->input('filter')) {
            case 'today':
                $start = Carbon::now();
                $start->hour = $start->minute = $start->second = 0;
                $previous_period_start = (clone $start)->subDay();
                $period_name = 'yesterday';
                break;

            case 'month':
                $start = Carbon::now()->subMonth();
                $previous_period_start = (clone $start)->subMonth();
                $period_name = 'previous month';
                break;

            case 'custom':
                $end = Carbon::createFromFormat('Y-m-d', $request->input('date_to'));
                $start = Carbon::createFromFormat('Y-m-d', $request->input('date_from'));
                $previous_period_start = (clone $start)->subDays($start->diffInDays($end));
                $period_name = 'previous period';
                break;

            default:
            case 'week':
                $start = Carbon::now()->subDays(7);
                $previous_period_start = (clone $start)->subDays(7);
                $period_name = 'previous week';
                break;
        }

        /*
         * Store the selected filter in session
         * That way we can ensure that page reload doesn't reset filter
         * this ignores custom filter
         */
        if ($request->input('filter') !== 'custom') {
            session()->put('admin.dashboard.filter', $request->input('filter'));
        }

        $stats = self::getStats($start, $end, $previous_period_start);
        return [
            'quick_stats' => $stats->quick_stats->sortBy('order')->values(),
            'charts' => $stats->charts ?? [],
            'period_name' => $period_name,
            'date_data' => $stats->cache_string
        ];
    }

    public function chart(Request $request)
    {
        // Disable debugbar on this view
        if (app()->bound('debugbar')) {
            app('debugbar')->disable();
        }

        $stats = Cache::tags('dashboard-stats')->get($request->input('date_data'));
        $chart = $stats->charts->where('id', $request->input('type'))->first();
        return view('core::admin.chart', [
            'chart' => $chart['data']
        ]);

    }

    /**
     * Get the dashboard stats to display
     *
     * @param Carbon $start
     * @param Carbon $end
     * @return array
     */
    public static function getStats(Carbon $start, Carbon $end, Carbon $previous_period_start)
    {
        $start_date = $start->format('Y-m-d');
        $end_date = $end->format('Y-m-d');
        $previous_period_date = $previous_period_start->format('Y-m-d');
        $cache_string = "admin-dashboard-{$start_date}-{$end_date}-{$previous_period_date}";

        if (config('core.cache_dashboard_stats', false)) {
            $stats = Cache::tags('dashboard-stats')->get($cache_string);
            if (!$stats) {
                $stats = collect(Event::fire(new FetchDashboardStats($start, $end, $previous_period_start)))->first();
                Cache::tags('dashboard-stats')->put($cache_string, $stats, 35);
            }
        } else {
            $stats = collect(Event::fire(new FetchDashboardStats($start, $end, $previous_period_start)))->first();
        }


        $stats->cache_string = $cache_string;
        return $stats;
    }


}
