<?php
/**
 * Controller for Users
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Response;
use Illuminate\Support\Facades\App;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Mtc\Core\Contracts\User;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Http\Requests\Admin\StoreUser;

/**
 * Controller for Users in Admin
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class UserController extends Controller
{
    /**
     * Ensure this is only accessible to those with the permission to do so.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:manage-users');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return (new Builder('core.admin.users', App::make(User::class)->query()))
            ->columns([
                'id' => 'ID',
                'full_name' => 'Name',
                'role' => 'Role',
            ])
            ->data([
                'role' => function ($user) {
                    return $user->roles->pluck('name')->implode(', ');
                }
            ])
            ->setDeletePermission('delete-users')
            ->view();
    }

    /**
     * Show the form for creating a new resource
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $user = App::make(User::class);

        return (new ItemBuilder('core.admin.users', $user))
            ->view('core::admin.user.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreUser $request Validated Request
     * @param int|bool $user_id id to user
     *
     * @return \Illuminate\Http\Response
     */
    public function store(StoreUser $request, $user_id = false)
    {
        $user = App::make(User::class)->findOrNew($user_id);

        $user->fill($request->all())
            ->save();

        $user->roles()->sync($request->input('roles', []));

        return redirect(route('core.admin.users.index'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param User $user User Model
     *
     * @return \Illuminate\View\View
     */
    public function edit($user_id)
    {
        $user = App::make(User::class)->findOrFail($user_id);
        return (new ItemBuilder('core.admin.users', $user))
            ->view('core::admin.user.form');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreUser $request Validated Request
     * @param int $user_id User to update
     *
     * @return \Illuminate\View\View|Response
     */
    public function update(StoreUser $request, $user_id)
    {
        return $this->store($request, $user_id);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request Incoming request
     * @param int $user_id User to Delete
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $user_id)
    {
        $user = App::make(User::class)->findOrFail($user_id);
        $success = false;
        if (stristr($user->email, '@mtcmedia.co.uk') === false) {
            $success = $user->delete();
        }

        if ($request->ajax() && $success === true) {
            return response('success', 200);
        } elseif ($request->ajax()) {
            return response('fail', 400);
        } elseif ($success === true) {
            $request->session()
                ->flash('success', "{$user->name} has been deleted.");
        } else {
            $request->session()
                ->flash('error', "{$user->name} has not been deleted.");
        }

        return redirect()->back();
    }
}
