<?php
/**
 * Class SeoComposer
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Http\ViewComposers;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Illuminate\View\View;
use Mtc\Core\Models\Seo\KnowledgeGraph;
use Mtc\Core\Models\Seo\PageMetaValues;
use Mtc\Core\Models\Seo\Setting;
use Mtc\Core\Node;
use Mtc\Core\Models\Seo\Defaults;

/**
 * Class SeoComposer
 *
 * Add in the Seo information for views
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class SeoComposer
{
    /**
     * Handle the view processing to add in required Seo fields
     *
     * @param View $view
     */
    public function compose(View $view)
    {
        /*
         * We need seo data only on main layout template.
         * To reduce duplicated content checks we discard all views that are
         * - not under layout namespace
         * - are partials under layout namespace
         * This will ensure that only main layout is loaded as long as template naming convention is followed.
         * This ends up being a large resource saving as each page will consist of at least 3 views
         * More advanced views will be even more advanced.
         */
        if (strpos($view->getName(), 'layout') === false || strpos($view->getName(), 'partials') !== false) {
            return;
        }

        // Load Meta author from Seo Settings
        $meta_author = Cache::remember('META_AUTHOR', 60, function () {
            return Setting::query()
                ->where('key', 'META_AUTHOR')
                ->first();
        });

        $view->with([
            'seo_data' => self::getSeoData($view),
            'seo_tags' => self::setSeoTags(),
            'meta_author' =>  $meta_author->value ?? '',
            'knowledge_graph' => File::exists(public_path(KnowledgeGraph::$graph_filename)) ? '/' . KnowledgeGraph::$graph_filename : ''
        ]);
    }

    /**
     * Generate Seo Data for current page
     *
     * @param View $view Current view to render
     * @return object
     */
    public static function getSeoData(View $view)
    {
        // Get the URL of the page
        $url = request()->getPathInfo();

        // Discard all admin side checks as admin is seo-invisible due to password protection
        if (strpos($url, 'admin') !== false ) {
            return false;
        }

        /*
         * Check if meta information is set for this url specifically
         * This is done from Page Metadata section in admin
         * This will only work for exact match urls
         */
        $url_meta = PageMetaValues::query()
            ->where('url', $url)
            ->first();
        if ($url_meta) {
            return $url_meta;
        }

        /*
         * Lets find the Node object for this URL
         * If node exists it means there could be a Default associated with this node
         */

        $node = Node::query()
            ->where('url', $url)
            ->first();

        /*
         * Check if we have defaults for this page set
         * If we have node we use matching by node (Page / Product etc.)
         * Alternatively we use loose URL matching to find a url that would work for this page
         */
        $defaults = $node ? Defaults::getNodeDefaults($node) : Defaults::getDefaultsFromUrl($view, $url);
        if ($defaults) {
            return $defaults;
        }

        // Final fallback to Site Config
        return (object)[
            'title' => config('app.name'),
            'description' => config('app.name')
        ];
    }

    /**
     * Register all necessary 3rd party scripts that need to be included on site.
     * This if meant for functionalities like google tag manager.
     * The result of this function is an array of head and body sections (where script should be included)
     * Each section may consist of an array which has template and "with" values.
     *
     * @return array of tags to include for seo/marketing purposes
     */
    public static function setSeoTags()
    {
        $seo_tags = [
            'head' => [], // Scripts that need to be in head
            'body' => []  // Scripts that need to be in body
        ];

        /*
         * Check if Google Tag Manager has been enabled.
         * If it is enabled, pass the tag manager for inclusion in template
         * Cache this value for 1 hour. If it is changed from admin it will be cleared
         */
        $google_tag_manager = Cache::remember('GTM_CONTAINER_ID', 60, function () {
            return Setting::query()
                ->where('key', 'GTM_CONTAINER_ID')
                ->where('value', '!=', '')
                ->first();
        });

        if ($google_tag_manager) {
            $seo_tags['body'][] = [
                'template' => 'core::public.seo.google_tag_manager',
                'with' => [
                    'GTM_CONTAINER_ID' => $google_tag_manager->value
                ]
            ];
        } else {
            /*
             * Google Analytics is a fall-back and should be used only when Tag Manager is not used
             * If it is enabled, pass the tag manager for inclusion in template
             * Cache this value for 1 hour. If it is changed from admin it will be cleared
             */
            $analytics = Cache::remember('GA_CODE', 60, function () {
                return Setting::query()
                    ->where('key', 'GA_CODE')
                    ->where('value', '!=', '')
                    ->first();
            });

            // If we don't have Analytics code defined in admin use the config fallback
            // The fallback should be the mtc test account
            $analytics_code = $analytics ? $analytics->value : config('core.seo_analytics_test_account');

            $seo_tags['body'][] = [
                'template' => 'core::public.seo.google_analytics',
                'with' => [
                    'GA_CODE' => $analytics_code
                ]
            ];
        }

        // Return all enabled seo tags
        return $seo_tags;
    }
}
