<?php
/**
 * Class Seo\KnowledgeGraph
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Models\Seo;

use Baum\Extensions\Eloquent\Collection;
use Illuminate\Support\Facades\File;

/**
 * Class Seo\KnowledgeGraph
 *
 * Seo Knowledge Graph model
 * Used to store knowledge graph fields and values
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class KnowledgeGraph extends \Baum\Node
{

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'seo_knowledge_graph';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'field',
        'type',
        'value'
    ];

    /**
     * @var string $graph_filename path to knowledge graph file
     */
    public static $graph_filename = 'js/knowledge_graph.json';

    /**
     * Boot the model
     */
    public static function boot()
    {
        parent::boot();

    }

    /**
     * Generate the full tree for Knowledge graph
     *
     * @return \Illuminate\Support\Collection
     */
    public static function renderTree()
    {
        return self::roots()
            ->get()
            ->keyBy('field')
            ->map(function (self $leaf) {
                return $leaf->processChildren();
            });
    }

    /**
     * Build up a recursive structure for Knowledge graph
     * This function recursively calls itself if nescessary
     *
     * @return mixed
     */
    public function processChildren()
    {
        if ($this->type === 'text') {
            return $this->value;
        } elseif ($this->type === 'list') {
            return $this->children->pluck('field')->toArray();
        } else {
            return $this->children
                ->keyBy('field')
                ->map(function (self $leaf) {
                    return $leaf->processChildren();
                });
        }
    }

    /**
     * Retrieve current saved knowledge tree from stored file.
     *
     * @return string JSON string of current Knowledge graph. Returns message if file does not exist
     */
    public static function currentPublicGraph()
    {
        if (!File::exists(public_path(self::$graph_filename))) {
            return 'File does not exist';
        }
        return File::get(public_path(self::$graph_filename));
    }

    /**
     * Generate Knowledge graph file.
     *
     * This method saves the knowledge graph to a file so it can be loaded on site
     */
    public static function generateGraph()
    {
        File::put(
            public_path(self::$graph_filename),
            json_encode(self::renderTree(), JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT)
        );
    }
}
