<?php

namespace Mtc\Core\AddressFinder\Drivers;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Jcf\Geocode\Geocode;
use Mtc\Core\Contracts\AddressFinder;
use Mtc\Core\Traits\AddressValidator;
use SKAgarwal\GoogleApi\PlacesApi;

/**
 * Class RoyalMailAddressFinder
 *
 * Class that implements address searching through Royal Mail Database
 * This database is Located on paf.mtcassets.com and can be accessed directly by mysql.
 *
 * @package Mtc/Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class GooglePlacesAddressFinder implements AddressFinder
{
    /**
     * Implement search functionality on Royal Mail
     *
     * @param string $query given search string - either postcode or partial address
     * @return Collection Collection of found addresses
     * @throws \Exception Exception about wildcard search
     */
    public function search($query)
    {
        try {
            $response = $this->find($query);
            return $this->prepareForDisplay($response->results);
        } catch (\Exception $exception) {
            return [];
        }
    }

    private function find($address)
    {
        $client = new \GuzzleHttp\Client();
        $params = [
            'address' => $address,
            'key' => env('GEOCODE_GOOGLE_APIKEY'),
            'language' => 'en'
        ];

        return json_decode($client->get('https://maps.googleapis.com/maps/api/geocode/json', [
            'query' => $params
        ])->getBody());

    }

    /**
     * Convert the collection of addresses to a render friendly format
     *
     * @param Collection $address_collection
     * @param string $query_postcode original postcode value to show search visibility
     * @return mixed
     */
    private function prepareForDisplay($address_list)
    {
        return collect($address_list)->map(function ($address) {

            $address1 = collect($address->address_components)
                ->filter(function ($component) {
                    return in_array('subpremise', $component->types)
                        || in_array('premise', $component->types)
                        || in_array('street_number', $component->types);
                })
                ->implode('long_name', ', ');
            $address2 = collect($address->address_components)
                ->filter(function ($component) {
                    return in_array('route', $component->types);
                })
                ->implode('long_name', ', ');

            $city = collect($address->address_components)
                ->filter(function ($component) {
                    return in_array('postal_town', $component->types);
                })
                ->first();

            $postcode = collect($address->address_components)
                ->filter(function ($component) {
                    return in_array('postal_code', $component->types);
                })
                ->first();

            $country = collect($address->address_components)
                ->filter(function ($component) {
                    return in_array('country', $component->types);
                })
                ->first();

            return collect([
                'id' => $address->place_id,
                'display_address' => $address->formatted_address,
                'address1' => $address1,
                'address2' => $address2,
                'city' => $city->long_name,
                'state' => '',
                'postcode' => $postcode->long_name,
                // Since this is a Royal mail DB all addresses are GB
                'country' => $country->short_name,

            ]);
        });
    }
}