<?php
/**
 * Command Line Install Helper
 *
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;

/**
 * Installer Class
 *
 * Provide a facility to create environment automatically.
 *
 * @package Mtc\Core\Console\Commands
 */
class Install extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'core:install';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Triggers install process';

    /**
     * Store the environment variables we'll be changing from the .env file.
     *
     * @var array
     */
    protected $env = [];

    /**
     * Create a new command instance
     */
    public function __construct()
    {
        parent::__construct();

        $user      = get_current_user();
        $key       = $user . '_' . strtolower(str_random(5));
        $this->env = [
            'DB_HOST'     => 'localhost',
            'DB_DATABASE' => $key,
            'DB_USERNAME' => $key,
            'DB_PASSWORD' => str_random(10) . '$',
        ];
    }

    /**
     * Execute the console command for generating a new certificate.
     *
     * @return mixed
     */
    public function handle()
    {
        $this->createEnv()
             ->createDatabase()
             ->updateEnv()
             ->migrate();
    }

    /**
     * Create the environment file (if it does not exist) and run the key:generate command.
     *
     * @return $this
     */
    protected function createEnv()
    {
        if (!file_exists($this->laravel->environmentFilePath())) {
            copy($this->laravel->environmentFilePath() . '.example', $this->laravel->environmentFilePath());
            Artisan::call('key:generate', [], $this->getOutput());
        }

        return $this;
    }

    /**
     * Create a new database automatically with cPanel.
     *
     * @return $this|Install
     */
    protected function createDatabase()
    {
        if (true == $this->confirm('Would you like to provide your own database details?', false)) {
            $this->askForDatabaseDetails();
        }

        if (false === $this->confirm('Would you like the database to be make automatically?', true)) {
            return $this;
        }

        $cpanel_password = $this->secret('cPanel Password');
        $context         = stream_context_create([
            'ssl' => [
                'allow_self_signed' => true,
                'verify_peer'       => false,
                'verify_peer_name'  => false,
            ],
        ]);
        $base_url        = 'https://'
                           . trim(`whoami`)
                           . ':'
                           . urlencode($cpanel_password)
                           . '@localhost:2083/execute/Mysql/';

        // Create Database
        file_get_contents(
            $base_url . 'create_database?name=' . urlencode($this->env['DB_DATABASE']),
            false,
            $context
        );
        $this->info("Created Database: " . $this->env['DB_DATABASE']);

        // Create User
        file_get_contents($base_url . 'create_user?' . http_build_query([
                'name'     => $this->env['DB_USERNAME'],
                'password' => $this->env['DB_PASSWORD'],
            ]), false, $context);

        $this->info("Created Username: " . $this->env['DB_USERNAME']);

        // Assign User to Database
        file_get_contents($base_url . 'set_privileges_on_database?' . http_build_query([
                'user'       => $this->env['DB_USERNAME'],
                'database'   => $this->env['DB_DATABASE'],
                'privileges' => 'ALL PRIVILEGES',
            ]), false, $context);

        $this->info("Assigned Privileges");

        return $this;
    }

    /**
     * Ask the user for database details rather than generating these automatically.
     *
     * @return $this
     */
    protected function askForDatabaseDetails()
    {
        $this->env = [
            'DB_HOST'     => $this->ask('MySQL Host', $this->env['DB_HOST']),
            'DB_DATABASE' => $this->ask('MySQL Database', $this->env['DB_DATABASE']),
            'DB_USERNAME' => $this->ask('MySQL Username', $this->env['DB_USERNAME']),
            'DB_PASSWORD' => $this->ask('MySQL Password', $this->env['DB_PASSWORD']),
        ];

        return $this;
    }

    /**
     * Update the .env file with the latest settings.
     *
     * @return $this
     */
    protected function updateEnv()
    {
        $file = file_get_contents($this->laravel->environmentFilePath());
        foreach ($this->env as $key => $value) {
            $file = preg_replace("/^{$key}=.*/m", "{$key}={$value}", $file);
        }
        file_put_contents($this->laravel->environmentFilePath(), $file);

        return $this;
    }

    /**
     * Run the artisan command to migrate and seed the database automatically.
     */
    protected function migrate()
    {
        $this->info(`php artisan migrate --seed`);
    }
}
