<?php
/**
 * CustomField Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\CustomFields;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;

/**
 * CustomField Eloquent Model
 *
 * Used to store data about a custom field so we know it exists and what it's
 * made out of.
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class CustomField extends Model
{
    /**
     * @var string[] $text_types Custom Field Types that use Text column instead of varchar column in DB
     */
    private static $text_types = [
        'text-area',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        // CustomGroup parent
        'group_id',
        // It's column name within the group's table
        'column_name',
        // It's type (text, textarea, wysiwyg, dropdown, etc)
        'type',
        // A taxonomy that should be used as options for this.
        'list_id',
        // Check if multiple options are selectable
        'multiple'
    ];

    /**
     * @var string[] cast attribute types
     */
    protected $casts = [
        'multiple' => 'boolean',
        'translatable' => 'boolean',
    ];

    /**
     * The relations which should have their timestamps updated if this model
     * is updated/created.
     *
     * @var array
     */
    protected $touches = ['group'];

    /**
     * The field's parent group.
     *
     * @return BelongsTo
     */
    protected function group()
    {
        return $this->belongsTo(CustomGroup::class);
    }

    /**
     * The related taxonomy/list for this field, if any.
     *
     * @return BelongsTo
     */
    protected function taxonomy()
    {
        return $this->belongsTo(Taxonomy::class, 'list_id')->with('node');
    }

    /**
     * Check if the custom field should be a text field or varchar field
     * Text fields are not index-able so we do not want to store data here if not necessary
     *
     * @return bool
     */
    public function isText()
    {
        return in_array($this->type, self::$text_types) || $this->translatable;
    }

    /**
     * Return the column name which should be updated based on content expectation
     *
     * @return string
     */
    protected function updateColumn()
    {
        return $this->isText() ? 'text_value' : 'value';
    }

    /**
     * Save custom field value for given node
     *
     * @param CustomField $field
     * @param $value
     * @throws \Exception
     */
    public function saveValue(Node $node, $value)
    {
        if ($this->multiple) {
            if (!is_array($value)) {
                throw new \Exception("Custom Field [{$this->title}] expects array when saving data");
            }

            // Delete entries that have been unselected
            CustomFieldValue::query()
                ->where('group_id', $node->group_id)
                ->where('field_id', $this->id)
                ->where('node_id', $node->id)
                ->whereNotIn($this->updateColumn(), $value)
                ->delete();

            // Loop through all selected values to make sure they are saved
            foreach ($value as $data_entry) {
                CustomFieldValue::query()
                    ->firstOrNew([
                        'group_id' => $node->group_id,
                        'field_id' => $this->id,
                        'node_id' => $node->id,
                        $this->updateColumn() => $data_entry
                    ])
                    ->save();
            }
            return;
        }

        $custom_value = CustomFieldValue::query()
            ->firstOrNew([
                'group_id' => $node->group_id,
                'field_id' => $this->id,
                'node_id' => $node->id,
            ]);

        /*
         * Here we fill the appropriate field
         * This can be either text_value or value field depending on necessary storage amount
         * e.g. Translatable fields as well as text-areas require storage in text block
         */
        $custom_value->fill([
            $this->updateColumn() => $value
        ]);
        $custom_value->save();

    }

}
