<?php
/**
 * Seo Admin PageController
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\Seo;

use Barryvdh\Debugbar\Facade as DebugBar;
use Illuminate\Support\Facades\Event;
use League\Csv\Reader;
use League\Csv\Writer;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Http\Requests\Admin\SeoPageMetaBulkUpload;
use Mtc\Core\Http\Requests\Admin\StoreSeoPageMetaValue;
use Mtc\Core\Models\Seo\PageMetaValues;

/**
 * Class Admin\Seo\PageController
 *
 * Control Page specific meta information.
 * This section allows setting page specific meta information for urls
 * These values include page title, description and h1 tag
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class PageController extends Controller
{
    /**
     * Construct controller
     *
     * Ensure this is only accessible to those with the 'manage-seo' permission.
     * Add bulk upload form to listing page
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-seo');
        /*
         * Listen to footer event for listing page.
         * We need to inject a form that will allow bulk upload of meta data
         */
        Event::listen('core.builder.core.admin.seo.page_meta_tags.foot', function () {
            return 'core::admin.seo.page_meta.bulk_upload';
        });
    }

    /**
     * Render a listing page showing all defined meta values on site
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $builder = (new Builder('core.admin.seo.page_meta_tags', PageMetaValues::query(), 'Page Metadata'))
            ->columns(
                [
                    'url' => trans('fields.url'),
                    'title' => trans('fields.title'),
                    'description' => trans('fields.description'),
                    'h1' => trans('fields.h1'),
                ]
            );
        $builder->title = 'Page Meta Values';
        return $builder->view();
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $default = new PageMetaValues();
        $resource_name = 'core.admin.seo.page_meta_tags';
        $action = route(
            "{$resource_name}.store",
            [$default->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.page_meta.form',
                'name' => $resource_name,
                'item' => $default,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Save Meta Values
     *
     * @param StoreSeoPageMetaValue $request Request with validation
     * @param PageMetaValues $page_meta Meta values object
     *
     * @return \Illuminate\Http\Response Redirect to seo default listing
     */
    public function store(StoreSeoPageMetaValue $request, PageMetaValues $page_meta)
    {
        $page_meta->fill($request->all())
            ->save();

        $request->session()->flash('success', 'Meta values updated for ' . $page_meta->url);
        return redirect(route('core.admin.seo.page_meta_tags.index'));
    }

    /**
     * Show a view to allow the admin to edit an existing page meta value instance.
     *
     * @param PageMetaValues $page_meta Meta values object
     *
     * @return \Illuminate\View\View
     */
    public function edit(PageMetaValues $page_meta)
    {
        $resource_name = 'core.admin.seo.page_meta_tags';
        $action = route(
            "{$resource_name}.update",
            [$page_meta->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.page_meta.form',
                'name' => $resource_name,
                'item' => $page_meta,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Save an existing page meta value instance.
     *
     * @param StoreSeoPageMetaValue $request Validated request
     * @param PageMetaValues $page_meta Meta values object
     *
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function update(StoreSeoPageMetaValue $request, PageMetaValues $page_meta)
    {
        return $this->store($request, $page_meta);
    }

    /**
     * Delete page meta information values for a specific url
     *
     * @param Request $request Normal request
     * @param PageMetaValues $page_meta Meta values object
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, PageMetaValues $page_meta)
    {
        $page_meta->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Page Meta values for '{$page_meta->url}' has been deleted.");
        return redirect()->back();
    }

    /**
     * Process bulk upload of meta values for pages.
     * Upload is done via CSV file.
     *
     * @param SeoPageMetaBulkUpload $request
     * @return \Illuminate\Http\Response
     */
    public function bulkUpload(SeoPageMetaBulkUpload $request)
    {
        /*
         * Since this is a temporary file and we don't need to store it we will just use the temp file.
         * File has already been validated via Request class so we can open it and process it
         */
        $file_name = $request->file('meta_csv')->getRealPath();

        // Load CSV file from path
        $csv_file = Reader::createFromPath($file_name);

        foreach ($csv_file->fetchAll() as $page_meta_row) {
            /*
             * We need to find records that have url specified in each row.
             * If this url already exits in the database the entry will be overwritten
             * Otherwise we will create a new record
             */
            $page_meta = PageMetaValues::firstOrCreate([
                'url' => $page_meta_row[0]
            ]);

            // Set title and description
            $page_meta->title = $page_meta_row[1];
            $page_meta->description = $page_meta_row[2];

            // h1 values are optional
            if (!empty($page_meta_row[3])) {
                $page_meta->h1 = $page_meta_row[3];
            }

            // Save meta values
            $page_meta->save();
        }

        $request->session()->flash('success', "Bulk upload file has been processed.");
        return redirect()->back();
    }

    /**
     * Generate an example CSV file that can be used as an example for bulk uploads
     * This uses an array of dummy data showing different combinations possible
     * Nothing is returned as $csv->output() streams a csv file with all relevant headers
     */
    public function bulkExampleDownload()
    {
        DebugBar::disable();

        // Set up example data export
        $example_csv_data = [
            [
                '/some-url',
                'Title',
                'Description',
                'H1'
            ],
            [
                '/lorem-ipsum',
                'Title',
                'Description',
                ''
            ],
            [
                '/lorem-ipsum-dolor-sit-amet',
                'Title',
                '',
                'Unique H1'
            ],
            [
                '/only-h1',
                '',
                '',
                'When we have multiple words we encase them in quotes'
            ],
        ];

        // Create CSV object
        $csv = Writer::createFromString('');

        // Fill with Example data
        $csv->insertAll($example_csv_data);

        // Output the csv file
        $csv->output('buk_export_example.csv');
    }
}
