<?php
/**
 * Details Controller Class
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@gmail.com>
 */

namespace Mtc\Core\Http\Controllers\Members;

use Illuminate\Support\Facades\Auth;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Country;
use Mtc\Core\Models\User\Address;

/**
 * Details Controller Class
 *
 * User details section where a person can manage their account details
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@gmail.com>
 */
class DetailsController extends Controller
{

    /**
     * DetailsController constructor.
     *
     * Adds requirement for logging in as middleware and adds breadcrumbs
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('auth');
        $this->breadcrumbs->addCrumb(
            ucwords(trans('core::fields.user_details.details')),
            action('\\' . self::class . '@index')
        );
    }

    /**
     * Display the details page where users can see and adjust their info.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $user = Auth::user();
        $addresses = $user->addresses;
        $countries = Country::getOrderedActiveCountries();
        return view('core::public.members.details')->with(compact('addresses', 'countries', 'user'));
    }

    /**
     * Update the user details.
     *
     * @param  \Illuminate\Http\Request $request
     * @param Address $address
     * @return \Illuminate\Http\Response
     * @internal param int $id
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        // make sure email is unique
        $validated = $this->validate($request, [
            'company_name' => 'max:255',
            'full_name' => 'required|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'contact_no' => 'required'
        ]);

        $user->fill($validated);

        // If user is changing password, validate password
        if ($request->input('change_password')) {
            $this->validate($request, [
                'password' => 'required|confirmed|password',
            ]);

            // In Laravel password is encrypted using bcrypt() method
            $user->password = bcrypt($request->input('password'));
        }

        $user->save();

        // set the success message and return back to index page
        session()->flash('success', trans('core::fields.information_updated'));
        return redirect(route('core.member.details.index'));
    }

}
