<?php
/**
 * Allow global search by looking through Node results.
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin;

use Auth;
use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Allow global search by looking through Node results.
 *
 * Used in the admin header to show a list of 10 closest results.
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class SearchController extends Controller
{
    /**
     * Perform a search request.
     *
     * @param Request $request Normal request
     * @return mixed Collection | Error string
     */
    public function search(Request $request)
    {
        if (Auth::user()->hasPermissionTo('admin')) {

            $search = $request->input('query', '');

            return Node::search($search)
                ->get()
                ->reject(function ($item) {
                    // if the item isn't nodeable remove it from the results
                    if (!$item->nodeable) {
                        return true;
                    }
                    // check the users related permissions, removing the item if necessary
                    if (Auth::user()->hasPermissionTo('manage-custom-field-taxonomies') &&
                        $item->nodeable_type === Taxonomy::class) {
                        return false;
                    }

                    if (Auth::user()->hasPermissionTo('manage-custom-fields') &&
                        $item->nodeable_type === Product::class) {
                        return false;
                    }

                    if (Auth::user()->hasPermissionTo('manage-custom-fields') &&
                        $item->nodeable_type === Variant::class) {
                        return false;
                    }

                    return true;
                })
                ->map(function ($item) {
                    // set the url of the item using it's related getAdminUrl function
                    $item->url = ($item->nodeable_type)::where('id', $item->nodeable_id)
                        ->get()
                        ->first()
                        ->getAdminUrl();
                    return $item;
                })
                ->take(10);
        } else {
            return response(403);
        }
    }
}
