<?php
/**
 * Media Controller
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Http\Controllers;

use Illuminate\Http\Request;
use Mtc\Core\Http\Requests;
use Mtc\Media\Media;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Media Controller
 *
 * This controller acts as a middleware that ensures not found images are not meant
 * to be generated on the fly. Missing image files are processed for requested size and rendered as a result.
 * If an image is found here it is also saved to ensure this resource is not hit every time
 *
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class MediaController extends Controller
{
    /**
     * Display uploaded resource
     *
     * Media management has "soft-cache" for images.
     * Image will be attempted to load through direct file access.
     * If image is not found a route is used to check if image exists.
     * Image sizes are generated on the fly.
     * This will auto-generate the image size to reduce the load that comes with bulk image upload & processing
     *
     * @throws NotFoundHttpException Exception if file not found
     * @throws \Exception Exception if file not found
     * @param string $storage_file file we are trying to access
     * @return mixed Resized image with all necessary headers
     */
    public function show($storage_file)
    {
        $path_info = pathinfo($storage_file);

        // No extension means this is not a file access
        if (empty($path_info['extension'])) {
            throw new NotFoundHttpException('Resource is not a file', null, 404);
        }

        /*
         * Check if this extension is an image allowed for image resizing
         * If this is a supported image we can attempt to resize lets proceed
         */
        if (in_array($path_info['extension'], Media::$supported_image_extensions)) {
            return $this->processImage($path_info);
        }

        // We didn't find anything, throw 404 for a missing file
        throw new NotFoundHttpException('File Not Found', null, 404);
    }

    /**
     * Process image based on given request information.
     * This attempts to find the image in database.
     * If image is found a size for it is generated.
     *
     * @param array $path_info
     * @throws NotFoundHttpException Image not found exception
     * @throws \Exception Image not found exception
     * @return mixed
     */
    private function processImage(array $path_info)
    {
        /**
         * @var Media $media Media file that has this image name
         */
        $media = Media::where('url_name', $path_info['basename'])
            ->firstOrFail();

        // return the generated media file
        return $media->generateImageSize($path_info['dirname']);
    }

    /**
     * Store request is used for storing a new image version
     * This is used by the image crop tool which sends the new coordinates for image size
     * Method finds the media record and attempts cropping the original image to specified size
     *
     * @param Request $request incoming request
     * @return string url to the cropped image
     */
    public function store(Request $request)
    {
        /** @var Media $media */
        $media = Media::findOrFail($request->input('media.id'));

        // Crop image based on given coordinates, receive url path in the end
        return $media->cropSize($request->input('size.path'), $request->input('coordinates'));


    }
}
