<?php
/**
 * Node Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core;

use DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Laravel\Scout\Searchable;
use Mtc\Core\CustomFields\CustomField;
use Mtc\Core\CustomFields\CustomFieldValue;
use Mtc\Core\CustomFields\CustomGroup;
use Mtc\Media\Media;
use Nicolaslopezj\Searchable\SearchableTrait;
use Spatie\Translatable\HasTranslations;

/**
 * The 'Node' is a cornerstone model that can connect to multiple different
 * models.
 *
 * It's mainly used to define a common set of fields, store a
 * unique slug which could later be used for access, and open access to
 * Custom Fields.
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Node extends Model
{
    use HasTranslations;
    use Searchable;

    const NODE_STATUS_PUBLISHED = 'published';
    const NODE_VISIBILITY_PUBLIC = 'public';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'slug',
        'description',
        'status',
        'featured',
        'visibility',
        'group_id',
        'custom_url',
        'url'
    ];

    /**
     * @var string[] columns to hide from output
     */
    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    /**
     * @var string[] list of fields that are translatable
     */
    public $translatable = [
        'title',
        'description'
    ];

    /**
     * The array which is serialized by Laravel Scout to provide text based
     * searching.
     *
     * @return array
     */
    public function toSearchableArray()
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            // 'description' => $this->description,
            'nodeable_type' => $this->nodeable_type
        ];
    }

    /**
     * Register a custom save event to create slugs and generate URLs.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::saving(
            function ($node) {

                if (empty($node->slug)) {
                    $node->createSlug(str_slug($node->title));
                } else {
                    $node->createSlug($node->slug);
                }

                if (!empty($node->nodeable)
                    && method_exists($node->nodeable, 'getUrl')
                    && $node->custom_url == 0
                ) {
                    $node->url = $node->nodeable->getUrl(false, $node);
                }
            }
        );
    }

    /**
     * Get a list of all owning nodeable models.
     *
     * @return MorphTo
     */
    public function nodeable()
    {
        return $this->morphTo();
    }

    /**
     * Retrive information about a custom group associated with this Node.
     *
     * @return BelongsTo
     */
    public function group()
    {
        return $this->belongsTo(CustomGroup::class, 'group_id');
    }

    /**
     * Retrieve the custom data for the current node.
     *
     * @return [type] [description]
     */
    public function getGroupDataAttribute()
    {
        $values = CustomFieldValue::query()
            ->ofGroup($this->group_id)
            ->ofNode($this->id)
            ->with([
                'field',
                'media'
            ])
            ->get();

        $data = [];
        foreach ($values as $row) {
            $value = $row->data_value;

            if ($row->field->type === 'image') {
                /** @var Media $value */
                $value = $row->media->first();
                optional($value)->setDefaultAttributes($row->field->multiple ? 'thumb' : 'small');
            }


            /** @var CustomFieldValue $row */
            if ($row->field->multiple) {
                $data[$row->field->id][] = $value;
            } else {
                $data[$row->field->id] = $value;
            }
        }
        $data['node_id'] = $this->id;

        return $data;
    }

    /**
     * Save custom data for the current node.
     *
     * @param array $data Incoming data
     *
     * @return boolean
     */
    public function saveGroupData(array $data)
    {
        try {
            // Run through data
            foreach ($data as $field_id => $value) {
                /** @var CustomField $field */
                $field = CustomField::query()->find($field_id);

                /*
                 * Skip over any fields that are not passing field_id
                 * or are referencing field that does not exist
                 * e.g. node_id can be passed as part of data array
                 */
                if (!is_int($field_id) || !$field) {
                    continue;
                }

                $field->saveValue($this, $value);
            }


            // Sync taxonomies directly to node_taxonomy for better filter performance
            $this->taxonomies()->sync(
            // Convert input array to a collection
                collect($data)
                    ->reject(function ($item, $key) {
                        // Find fields that have this ID and have list_id
                        $field = $this->group
                            ->fields
                            ->where('id', $key)
                            ->where('list_id', '>', 0)
                            ->first();

                        // If field with ID of $key was not a taxonomy list, reject it
                        return empty($field) || empty($field->list_id);
                    })
                    // Flatten to 1D and convert to array
                    ->flatten()
                    ->toArray()
            );
        } catch (\Exception $exception) {
            return false;
        }

        return true;
    }

    /**
     * Create a unique slug for the current node, set in $this->slug
     *
     * @param string $slug Slug to match against.
     *
     * @return void
     */
    public function createSlug($slug)
    {
        $base = $slug;
        $index = 1;

        // Check to see if a slug exists. If it does, then we start
        // adding an index until it's unique. E.g. hats, hats-1, hats-2.
        while ($this->slugExists($slug) == false) {
            $slug = $base . '-' . $index;
            $index++;
        }

        $this->slug = $slug;
    }

    /**
     * Determine if a slug already exists matching the provided string, except
     * for the current node.
     *
     * @param string $slug Slug to compare
     *
     * @return bool
     */
    public function slugExists($slug)
    {
        return self::whereSlug($slug)
                ->where('id', '!=', $this->id)
                ->count() == 0;
    }

    /**
     * Get the related taxonomies through the node_taxonomy pivot table
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function taxonomies()
    {
        return $this->belongsToMany(Taxonomy::class);
    }

    /**
     * Provide a public scope to ensure that the node is published and
     * available to the public.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePublic($query)
    {
        return $query->whereStatus(self::NODE_STATUS_PUBLISHED)
            ->whereVisibility('public');
    }
}
