<?php

namespace Mtc\Core\Providers;

use Illuminate\Support\Facades\Validator;
use Illuminate\Support\ServiceProvider;
use Mtc\Core\Country;
use Mtc\Core\Traits\AddressValidator;
use Schema;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        Schema::defaultStringLength(191);

        // add validation rule named 'alpha_dash_slash'
        Validator::extend('alpha_dash_url', function ($attribute, $value, $parameters, $validator) {
            //allow all alpha-numeric characters with slashes and dashes
            return preg_match("#^/[a-zA-Z0-9_/-]+$#", $value);
        });

        Validator::replacer('alpha_dash_url', function ($message, $attribute, $rule, $parameters) {
            return "The $attribute field must contain alpha-numeric values with slashes and dashes. "
                . "It must start with /";
        });

        // add validation rule named 'alpha_dash_slash'
        Validator::extend('alpha_dash_file', function ($attribute, $value, $parameters, $validator) {
            //allow all alpha-numeric characters with slashes and dashes
            return preg_match("#^[a-zA-Z0-9_-]+\.[a-zA-Z]{3,4}$#", $value);
        });

        Validator::replacer('alpha_dash_file', function ($message, $attribute, $rule, $parameters) {
            return "File must be a filename that consists of alpha-numeric values"
                . " with slashes and/or dashes.";
        });

        // add validation rule named 'postcode'
        Validator::extendImplicit('postcode', function ($attribute, $value, $parameters, $validator) {
            // get country regex string & postcode status from database
            $country = Country::query()
                ->where('code', $parameters[0])
                ->get([
                    'code',
                    'has_postcodes',
                    'postcode_regex'
                ])
                ->first();

            if($country->has_postcodes) {
                return preg_match("#" . $country->postcode_regex . "#", AddressValidator::normalizePostcode($value, $country->code));
            }
            return true;
        });

        Validator::replacer('postcode', function ($message, $attribute, $rule, $parameters) {
            return "Invalid Postcode";
        });

        /*
         * Add filter with name multiple_emails
         * This filter validates multiple email addresses
         * separated by a comma.
         */
        Validator::extendImplicit('multiple_emails', function ($attribute, $value, $parameters, $validator) {
            return collect(explode(',', $value))
                ->map('trim')
                ->reject(function ($email_address) {
                    return filter_var($email_address, FILTER_VALIDATE_EMAIL);
                })
                ->isEmpty();
        });

        Validator::replacer('multiple_emails', function ($message, $attribute, $rule, $parameters) {
            return "Email Addresses are not valid";
        });


        // add validation rule named 'password'
        Validator::extendImplicit('password', function ($attribute, $value, $parameters, $validator) {
            return (mb_strlen($value) > 7 // at least 8 chars
                && preg_match('/[0-9]/', $value) // at least number
                && preg_match('/[a-z]/i', $value)// at least one letter
            );
        });

        Validator::replacer('password', function ($message, $attribute, $rule, $parameters) {
            return "Insecure Password - must be at least 8 symbols and must contain both letters and numbers";
        });
    }

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        //
    }
}
