<?php
/**
 * Class UpdateCurrencies
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-27
 */
namespace Mtc\Core\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Mtc\Core\Models\Currency;

/**
 * Class UpdateCurrencies
 *
 * This script updates currencies from fixer api
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-27
 */
class UpdateCurrencies extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'currencies:update';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Connects to an API to update currency with latest ratios';

    /**
     * API_ENDPOINT points to the currency API from where rates are fetched
     */
    const API_ENDPOINT = 'http://api.fixer.io/latest';

    /**
     * Execute the console command for generating a new certificate.
     *
     * @return mixed
     */
    public function handle()
    {
        try {
            $rates = self::getCurrencyRates();

            // If failed to fetch rates throw an exception
            if (empty($rates)) {
                throw new \Exception('Failed to get Currency Rates');
            }

            // Loop through rates and update accordingly
            foreach ($rates as $code => $rate) {
                Currency::where('code', $code)
                    ->update([
                        'rate' => $rate
                    ]);
            }
        } catch (\Exception $exception) {
            // send notification to developer about error encountered
            Mail::send('core::emails.notification', [
                'content' => 'Currency update failed on ' . config('app.name')
            ], function ($mailer) {
                $mailer->from(config('mail.from.address'), config('mail.from.name'));
                $mailer->to(config('app.developer_email'))
                    ->subject('Currency update failed on ' . config('app.name'));
            });
        }
    }

    /**
     * Make the call to get rates from the API
     * The rate result must return the rates in format of $code => $rate
     * Rates must be calculated against default currency
     * This method is called in the artisan command as well as in Currency admin section
     *
     * @return float[] array in format $code => $rate
     */
    public static function getCurrencyRates()
    {
        $full_url = self::API_ENDPOINT . '?base=' . config('core.default_currency', 'GBP');
        $result = json_decode(file_get_contents($full_url), true);

        return !empty($result) ? $result['rates'] : [];
    }

}