<?php
/**
 * Class CurrencyController
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Http\Controllers\Admin;

use Mtc\Core\Console\Commands\UpdateCurrencies;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Models\Currency;

/**
 * Class CurrencyController
 *
 * Manage currencies in admin
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-30
 */
class CurrencyController extends Controller
{
    /**
     * Ensure this is only accessible to those with the permission to do so.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-currencies');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        // Set the ordering - check enabled first, then order by currency code
        $query = Currency::query()
            ->orderBy('active', 'desc')
            ->orderBy('code', 'asc');

        return (new Builder('core.admin.currencies', $query, trans('core::fields.currency_title')))
            ->columns([
                'code' => trans('core::fields.currency_code'),
                'rate' => trans('core::fields.currency_rate'),
                'active' => trans('core::fields.is_active')
            ])
            ->data([
                'active' => function ($currency) {
                    return $currency->active ? trans('core::fields.yes') : trans('core::fields.no');
                }
            ])
            ->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param Currency $currency currency model
     * @return \Illuminate\View\View
     */
    public function create(Currency $currency)
    {
        $resource_name = 'core.admin.currencies';
        $action = route(
            "{$resource_name}.store",
            [$currency->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.currencies.form',
                'name' => $resource_name,
                'item' => $currency,
                'node' => null,
                'supported_currencies' => array_keys(UpdateCurrencies::getCurrencyRates()),
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request Incoming Request
     * @param Currency $currency Currency Model
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Currency $currency)
    {
        $this->validate($request, [
            'code' => 'required|unique:currencies,code',
            'locale' => 'required',
            'active' => 'required|numeric'
        ]);

        // fetch the rate from API to make sure that don't enable currency with wrong ratio
        $all_rates = UpdateCurrencies::getCurrencyRates();

        $currency->fill($request->all());
        $currency->rate = $all_rates[$currency->code];
        $currency->save();

        $request->session()->flash('success', "Currency '{$currency->code}' has been added.");
        return redirect(action('\\' . self::class . '@index'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Currency $currency currency model
     * @return \Illuminate\View\View
     */
    public function edit(Currency $currency)
    {
        $resource_name = 'core.admin.currencies';
        $action = route(
            "{$resource_name}.update",
            [$currency->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.currencies.form',
                'name' => $resource_name,
                'item' => $currency,
                'node' => null,
                'supported_currencies' => array_keys(UpdateCurrencies::getCurrencyRates()),
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request Incoming Request
     * @param Currency $currency Currency to update
     * @return \Illuminate\View\View
     */
    public function update(Request $request, Currency $currency)
    {
        $this->validate($request, [
            'locale' => 'required',
            'active' => 'required|numeric'
        ]);

        $currency->fill($request->all());
        $currency->save();

        $request->session()->flash('success', "Currency '{$currency->code}' has been updated.");
        return redirect(action('\\' . self::class . '@index'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request Incoming Request
     * @param Currency $currency Currency Model
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Currency $currency)
    {
        $currency->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Currency '{$currency->code}' has been deleted.");
        return redirect()->back();
    }
}
