<?php
/**
 * Class Currencies
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-30
 */
namespace Mtc\Core\Http\Middleware;

use Closure;
use Mtc\Core\Models\Currency;

/**
 * Class Currencies
 *
 * Middleware for setting the correct currency locale on server.
 * This allows rendering the correct currency format for prices.
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-30
 */
class Currencies
{

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  Closure $next
     * @return Closure Next middleware to process
     */
    public function handle($request, Closure $next)
    {
        // If only default currency is used
        if (!config('core.currencies_enabled')) {
            // We need to make sure that the default currency is set correctly.
            // This prevents a bug when we switch currency off after leaving the currency to a non-default value
            // This can cause issues in places like ProductRepository which adjusts price within query instead of
            // collection
            if (session('currency_selected') !== config('core.default_currency')) {
                session()->put(['currency_selected' => config('core.default_currency')]);
            }
        }

        // If session doesn't have a currency set, fall back to default currency
        if (!session('currency_selected')) {
            session()->put('currency_selected', config('core.default_currency'));
        }

        /*
         * If the current url is from admin panel we need to show the default locale on all costs
         * We also need to go back to default locale when currencies are disabled
         * Otherwise we can check which currency is loaded for user and set that currency for locale
         * Monetary locale sets the currency symbol used on @currency() blade directive
         */
        if (config('core.currencies_enabled')
            && strpos(request()->route()->getPrefix(), config('core.admin_prefix')) === false
        ) {
            // Retrieve currency object
            $currency = Currency::getCachedCurrencyByCode(session('currency_selected'));

            // If currency object is found, set the locale
            if ($currency) {
                setlocale(LC_MONETARY, $currency->locale);
                return $next($request);
            }
        }

        // Fall back to the default currency locale when currency was not found or not relevant
        setlocale(LC_MONETARY, config('core.fallback_currency_locale'));
        return $next($request);
    }
}
