<?php
/**
 * Media Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Spatie\ImageOptimizer\Optimizers\Optipng;
use Spatie\ImageOptimizer\Optimizers\Pngquant;
use Spatie\ImageOptimizer\Optimizers\Jpegoptim;
use Spatie\ImageOptimizer\OptimizerChain;

/**
 * Media Eloquent Model
 *
 * Central location to store information about uploaded media.
 *
 * @package  Mtc\Core
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Media extends Model
{

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'type',
        'src',
        'url_name',
        'parent_id',
        'parent_type',
    ];

    /**
     * Define image types we support for upload/display
     * This is used when 404 parses missing image requests
     * TODO: This also will restrict image uploads
     *
     * @var string[] $supported_image_extensions
     */
    public static $supported_image_extensions = [
        'jpg',
        'jpeg',
        'png',
        'gif',
        'svg'
    ];

    /**
     * Some images like GIFs and SVGs can't be re-sized, for these we return the raw image
     * @var string[] File mimes for images that cannot be re-sized
     */
    public static $raw_image_formats = [
        'image/svg+xml', // SVG
        'image/gif', // GIF
    ];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        self::deleting(function (self $media) {
            // When deleting media we need to remove all its occurences

            // Find all sizes for this media through config
            $all_media_sizes = config('media.' . get_class($media->parent));

            // Loop through sizes
            foreach ($all_media_sizes as $size) {
                // If size has path (excludes original image storage path)
                if (!empty($size['path'])) {
                    // Delete this size
                    Storage::disk('public')->delete($size['path'] . '/' . $media->url_name);
                }
            }

            // Delete the raw source of the image
            Storage::disk('public')->delete($media->src);
        });

        self::saved(function (self $media) {
            if (array_key_exists('src', $media->getDirty())) {
                self::optimizeImage(Storage::disk('public')->path($media->src));
            }
        });

        self::saving(function (self $media) {
            // If we are uploading a new file we want to exclude any changes that relate to deleting or moving files
            // as there won't be any files to move created at this point
            $from_name = $media->getOriginal('url_name');
            if (empty($from_name)) {
                return;
            }

            // If src has changed, the sizes need to be cleared as different image was uploaded
            if (array_key_exists('src', $media->getDirty())) {
                // Find all sizes for this media through config
                $all_media_sizes = config('media.' . get_class($media->parent));

                // Loop through sizes
                foreach ($all_media_sizes as $size) {
                    // If size has path (excludes original image storage path)
                    if (!empty($size['path'])) {
                        if (Storage::disk('public')->exists($size['path'] . '/' . $from_name)) {
                            Storage::disk('public')->delete($size['path'] . '/' . $from_name);
                        }
                    }
                }
            }

            /*
             * Some admin users have permission 'manage-media-meta'
             * This permission allows to change the display name of the file
             * to allow better file naming.
             *
             * To do this we need to confirm that this file won't have the
             * original url name value as the original size image.
             * We also need to confirm that the url name has changed and we are about
             * to overwrite it with a new value.
             *
             * If the conditions are matched we loop through media sizes for the media object and
             * move the file around so the cropped file is moved also and we do not end up
             * with a newly generated file with different crop positioning
             */
            if (array_key_exists('url_name', $media->getDirty())) {

                // Find all sizes for this media through config
                $all_media_sizes = config('media.' . get_class($media->parent));

                // Loop through sizes
                foreach ($all_media_sizes as $size) {
                    // If size has path (excludes original image storage path)
                    if (!empty($size['path'])) {
                        $from_path = $size['path'] . '/' . $from_name;

                        /*
                         * If the size has already been generated in past
                         * Move it to a new location based on the new file name
                         */
                        if (Storage::disk('public')->exists($from_path)) {
                            Storage::disk('public')->move($from_path, $size['path'] . '/' . $media->url_name);
                        }
                    }
                }
            }
        });
    }

    /**
     * Set the default attributes for media file
     * This way we always will have consistent values for the url to media file, thumb loaded etc.
     */
    public function setDefaultAttributes()
    {
        // Set the URL of the image based on its location in the filesystem
        $this->href = Storage::disk('public')->url($this->src);
        // Get the thumb size
        $this->thumb_url = $this->getSize('thumb');
        // Check if this media can be resized
        $this->can_resize = !in_array($this->type, Media::$raw_image_formats);
    }

    /**
     * Image optimizer functionality to reduce the size of the image
     * Uses Spatie\ImageOptimizer functionality as base with few tweaks for our server specifics
     *
     * @param string $file_path path to media file that has to be optimized
     */
    public static function optimizeImage($file_path)
    {
        // Only proceeed if this is a supported image file we want to process
        if (!in_array(pathinfo($file_path, PATHINFO_EXTENSION), self::$supported_image_extensions)) {
            return;
        }

        // Jpegoptim is used for compressing jpeg/jpg files
        $jpeg_optim = new Jpegoptim([
            '--strip-all',
            '--all-progressive',
        ]);
        $jpeg_optim->binaryName = '/usr/local/bin/jpegoptim';

        // PNGQuant is used to compress image (lossy compression)
        $png_quant = new Pngquant([
            '--force',
        ]);
        $png_quant->binaryName = '/usr/local/bin/pngquant';

        // Optipng is a losless compression method
        $opti_png = new Optipng([
            '-i0',
            '-o2',
            '-quiet',
        ]);
        $opti_png->binaryName = '/usr/local/bin/optipng';

        // Create and execute optimizer chain on image
        $optimizer_chain = (new OptimizerChain())
            ->addOptimizer($jpeg_optim)
            ->addOptimizer($png_quant)
            ->addOptimizer($opti_png);
        $optimizer_chain->optimize($file_path);
    }

    /**
     * Get a list of all owning parent models.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function parent()
    {
        return $this->morphTo();
    }

    /**
     * Get URL of the original size for the uploaded media file
     *
     * @return string
     */
    public function getUrlAttribute()
    {
        return asset('storage/', $this->src);
    }

    /**
     * Generate the URL for size so we don't have to show full size images everywhere
     * Fallback to full size if size or Media owner does not exist
     *
     * @param string $size_name Size name to fetch
     * @param bool $storage_path whether to load url from beginning or without storage path
     * @throws NotFoundHttpException Size does not exist
     * @return string url for the image
     */
    public function getSize($size_info, $storage_path = true)
    {
        // Gif images will
        if (in_array($this->type, self::$raw_image_formats)) {
            return $this->src;
        }

        // Find all sizes for this media through config
        $all_media_sizes = config('media.' . get_class($this->parent));

        // If the config for this class doesn't exist or the size is not defined
        if (!$all_media_sizes) {
            throw new NotFoundHttpException('Media sizes for this object does not exist', null, 404);
        }

        // if $size_info is an array of width & height
        if (is_array($size_info) && count($size_info) == 2) {
            // Find size that matches the width & height values
            $size = collect($all_media_sizes)
                ->where('width', $size_info[0])
                ->where('height', $size_info[1])
                ->first();


            if (!$size) {
                // Size is not found in whitelisted images. Send back this path as it will 404
                return $size_info . '/' . $this->url_name;
            }

            // return image path with image size in pixels
            $image_path = pathinfo($size['path'])['dirname'] . '/' . $size['width'] . 'x' . $size['height'];
            return $image_path . '/' . $this->url_name;
        }

        // filter selected size
        $size = collect($all_media_sizes)
            ->filter(function ($size) use ($size_info) {
                // We need to make sure that size is an array and its last part matches selected size
                return is_array($size) && pathinfo($size['path'])['basename'] === $size_info;
            })->first();

        if (!$size) {
            // Size is not found in whitelisted images. Send back this path as it will 404
            return $size_info . '/' . $this->url_name;
        }

        $prefix = $storage_path ? '/storage/' : '';

        // Return the correct url of the image by replacing the original path with media path
        return $prefix . $size['path'] . '/' . $this->url_name;

    }


    /**
     * Attempt to generate a size based on a given path.
     * Will throw 404 exception if sizes are not available
     * This uses fit approach which crops and re-sizes based on given dimensions
     *
     * @param string $size_path Size access path
     * @throws NotFoundHttpException Size does not exist
     * @throws \Exception method was not able to set up a path for saving file
     * @return mixed
     */
    public function generateImageSize($size_path)
    {
        $size = $this->getSizeFromPath($size_path);

        // Size was not found also by dimensions
        if (!$size) {
            throw new NotFoundHttpException('This media size does not exist', null, 404);
        }

        $manager = new ImageManager([
            'driver' => config('core.image_manager_driver')
        ]);

        // Make sure we can upload to the required path
        $destination_path = public_path("storage/$size_path/");
        if (!is_dir($destination_path) && !File::makeDirectory($destination_path, 0775, true)) {
            throw new \Exception('Image destination path is not valid');
        }

        // Create image from original
        $image = $manager->make(public_path('storage/' . $this->src));

        // Resize to required size. Will be Square if height not provided
        $image->fit($size['width'], $size['height'] ?? null);

        // Save image to sizes path
        $image->save($destination_path . $this->url_name);

        self::optimizeImage($destination_path . $this->url_name);
        // Return image with relevant HTTP headers
        return $image->response();
    }

    /**
     * Crop image size based on given coordinates
     *
     * @param string $size_path Size which to crop
     * @param string[] $coordinates array off coordinates to crop
     * @return string cropped image url
     * @throws \Exception method was not able to set up a path for saving file
     */
    public function cropSize(string $size_path, array $coordinates)
    {
        /*
         * Coordinates are defined as x1, y1, x2, y2
         * We need to calculate the width & height for the crop area
         * Scaling down will be handled afterwards
         */
        $width = $coordinates[2] - $coordinates[0];
        $height = $coordinates[3] - $coordinates[1];

        $size = $this->getSizeFromPath($size_path);

        // Initialize image manager
        $manager = new ImageManager([
            'driver' => config('core.image_manager_driver')
        ]);

        $public_path = "/storage/$size_path/";

        // Make sure we can upload to the required path
        $destination_path = public_path($public_path);
        if (!is_dir($destination_path) && !File::makeDirectory($destination_path, 0775, true)) {
            throw new \Exception('Image destination path is not valid');
        }

        // Create image from original
        $image = $manager->make(public_path('storage/' . $this->src));

        // Pass the width, height and top-left corner coordinates to the crop tool
        $image->crop($width, $height, $coordinates[0], $coordinates[1]);

        // Resize to required size. Will be Square if height not provided
        $image->resize($size['width'], $size['height'] ?? $size['width']);

        // Save image to sizes path
        $image->save($destination_path . $this->url_name);

        // Return image url with timestamp for cache busting
        return $public_path . $this->url_name . '?' . time();

    }

    /**
     * Get the size array from given size path.
     * Function checks and returns given size config for this media object.
     *
     * @throws NotFoundHttpException Size was not found
     * @param string $size_path name of the size path
     * @return string[] size config array
     */
    private function getSizeFromPath($size_path)
    {

        // Find all sizes for this media through config
        $all_media_sizes = config('media.' . get_class($this->parent));

        // If the config for this class doesn't exist or the size is not defined
        if (!$all_media_sizes) {
            throw new NotFoundHttpException('This media size does not exist', null, 404);
        }

        // Convert to collection so we can filter and treat it as
        $size = collect($all_media_sizes)
            ->where('path', $size_path)
            ->first();

        /* Size was not found by name, lets check if name was a dimension
         * Regex checks the requested size to match something/123x321
         * This way we can change the image size dynamically during development:
         * - change the image ratio in config
         * - reload image in a template that has dimensions passed instead of size name
         */
        if (!$size && preg_match('#[-a-zA-Z]+/([0-9]+)x([0-9]+)#', $size_path, $matches)) {
            $size = collect($all_media_sizes)
                ->where('width', (int)$matches[1])
                ->where('height', (int)$matches[2])
                ->first();
        }
        return $size;
    }
}
