<?php
/**
 * Knowledge Graph Tests
 *
 * @category Mtc\Core\Tests
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

use Illuminate\Foundation\Testing\DatabaseTransactions;
use Mtc\Core\Http\Controllers\Admin\Seo\KnowledgeGraphController;
use Mtc\Core\Models\Seo\KnowledgeGraph;

/**
 * Knowledge Graph Tests
 *
 * These are the tests for confirming the knowledge
 * functionality is working without any unexpected issues
 *
 * @category Mtc\Core\Tests
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class KnowledgeGraphTest extends \Tests\TestCase
{
    use DatabaseTransactions;

    /**
     * Test that the creation and value set of text type element doesn't crash
     */
    public function testAddTextEntry()
    {
        $faker = Faker\Factory::create();
        $instance = KnowledgeGraph::create([
            'field' => $faker->colorName,
            'type' => 'text'
        ]);

        $instance->value = $faker->name;
        $instance->save();

        $this->assertInstanceOf(KnowledgeGraph::class, $instance);
    }

    /**
     * Test that the creation and value set of text type element doesn't crash
     */
    public function testAddListEntry()
    {
        $faker = Faker\Factory::create();
        $instance = KnowledgeGraph::create([
            'field' => $faker->colorName,
            'type' => 'list'
        ]);

        $max = random_int(1, 4);
        for ($i = 0;  $i < $max; $i++) {
            $child = $instance->children()->create([
                'field' => $faker->colorName,
                'type' => 'text'
            ]);

            $this->assertInstanceOf(KnowledgeGraph::class, $child);
        }

        $this->assertInstanceOf(KnowledgeGraph::class, $instance);

    }

    /**
     * Test that the creation and value set of object type element doesn't crash
     */
    public function testAddObjectEntry()
    {
        $faker = Faker\Factory::create();
        $instance = KnowledgeGraph::create([
            'field' => $faker->colorName,
            'type' => 'object'
        ]);

        $max = random_int(1, 4);
        for ($i = 0;  $i < $max; $i++) {
            $child = $instance->children()->create([
                'field' => $faker->colorName,
                'type' => 'text'
            ]);

            $child->field = $faker->name;
            $child->save();

            $this->assertInstanceOf(KnowledgeGraph::class, $child);
        }

        $this->assertInstanceOf(KnowledgeGraph::class, $instance);
    }

    /**
     * Test that preview page works and returns current tree
     */
    public function testPreview()
    {
        $this->actingAs(\Mtc\Core\Auth\User::role('mtc')->first())
            ->get(route('core.admin.seo.knowledge_graph.preview'))
            ->assertStatus(200)
            ->assertJson(KnowledgeGraph::renderTree()->toArray());
    }

    /**
     * Test that graph generation process works without issues
     */
    public function testGenerate()
    {
        // Trigger generate graph
        KnowledgeGraph::generateGraph();

        // Get the graph json file
        $graph = KnowledgeGraph::currentPublicGraph();

        // Get the current tree in DB for reference
        $tree = KnowledgeGraph::renderTree();

        // assert that graph is json
        $this->assertJson($graph);

        // assert that data structures match
        $this->assertJsonStringEqualsJsonString($tree->toJson(), $graph);

    }

    /**
     * Test that user is able to open knowledge graph manage page without errors
     */
    public function testViewAdminIndexPage()
    {
        $this->actingAs(\Mtc\Core\Auth\User::role('mtc')->first())
            ->get(action('\\' . KnowledgeGraphController::class . '@index'))
            ->assertStatus(200);
    }

}
