<?php
/**
 * Control admin actions over user roles.
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin;

use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Auth\Role;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Http\Requests\Admin\StoreRole;

/**
 * Control admin actions over user roles.
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class RoleController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-roles'
     * permission.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-roles');
    }

    /**
     * Render a browse page showing all the roles available in the system.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return (new Builder('core.admin.roles', Role::query()))
            ->columns(
                [
                'id' => trans('fields.id'),
                'name' => trans('fields.name'),
                ]
            )
            ->view();
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return (new ItemBuilder('core.admin.roles', new Role))
            ->view('core::admin.role.form', [
                'title' => trans('core::fields.user_role')
            ]);
    }

    /**
     * Create a new Role
     *
     * @param StoreRole $request Request with validation
     * @param Role      $role    Empty user role model
     *
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function store(StoreRole $request, Role $role)
    {
        $role->fill($request->all())
            ->save();

        $role->permissions()->sync($request->input('permissions', []));

        return redirect(route('core.admin.roles.index'));
    }

    /**
     * Show the current Role as JSON
     *
     * @param Role $role User Role
     *
     * @return \Illuminate\Http\Response JSON Response
     */
    public function show(Role $role)
    {
        return $role;
    }

    /**
     * Show a view to allow the admin to edit an existing role.
     *
     * @param Role $role Existing user role
     *
     * @return \Illuminate\View\View
     */
    public function edit(Role $role)
    {
        return (new ItemBuilder('core.admin.roles', $role))
            ->view('core::admin.role.form', [
                'title' => trans('core::fields.user_role')
            ]);
    }

    /**
     * Save an existing user role.
     *
     * @param StoreRole $request Validated request
     * @param Role      $role    Existing user role
     *
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function update(StoreRole $request, Role $role)
    {
        return $this->store($request, $role);
    }

    /**
     * Delete an existing role. If AJAX, we will send the OK. If not, we flash
     * a success message.
     *
     * @param Request $request Normal request
     * @param Role    $role    Existing user role
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Role $role)
    {
        $role->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Role '{$role->name}' has been deleted.");
        return redirect()->back();
    }
}
