<?php
/**
 * Seo Admin Settings Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin\Seo;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Http\Requests\Admin\StoreSeoSetting;
use Mtc\Core\Models\Seo\Setting;

/**
 * Class Seo\SettingsController
 *
 * Control Seo Settings used on site.
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class SettingsController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-seo'
     * permission.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-seo');
    }

    /**
     * Render a browse page showing all the defaults available in the system.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return (new Builder('core.admin.seo.settings', Setting::query(), 'Seo Settings'))
            ->columns([
                'key' => trans('fields.key'),
                'value' => trans('fields.value'),
                'description' => trans('fields.description'),
            ])
            ->view();
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $setting = new Setting();
        $resource_name = 'core.admin.seo.settings';
        $action = call_user_func_array(
            'route', [
                "{$resource_name}.store",
                [$setting->id]
            ]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.settings.form',
                'name' => $resource_name,
                'item' => $setting,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Create a new Setting
     *
     * @param StoreSeoSetting $request Request with validation
     * @param Setting $setting Existing seo setting instance
     * @return \Illuminate\Http\Response Redirect to seo setting listing
     */
    public function store(StoreSeoSetting $request, Setting $setting)
    {
        $setting->fill($request->all())
            ->save();

        $request->session()->flash('success', 'Setting updated for ' . $setting->key);
        return redirect(route('core.admin.seo.settings.index'));
    }

    /**
     * Show a view to allow the admin to edit an existing seo default instance.
     *
     * @param Setting $setting Existing seo setting instance
     * @return \Illuminate\View\View
     */
    public function edit(Setting $setting)
    {
        $resource_name = 'core.admin.seo.settings';
        $action = call_user_func_array(
            'route', [
                "{$resource_name}.update",
                [$setting->id]
            ]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'core::admin.seo.settings.form',
                'name' => $resource_name,
                'item' => $setting,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Save an existing seo setting instance.
     *
     * @param StoreSeoSetting $request Validated request
     * @param Setting $setting Existing seo setting instance
     *
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function update(StoreSeoSetting $request, Setting $setting)
    {
        return $this->store($request, $setting);
    }

    /**
     * Delete an existing seo setting instance
     *
     * @param Request $request Normal request
     * @param Setting $setting Existing seo setting instance
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Setting $setting)
    {
        $setting->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Seo setting '{$setting->key}' has been deleted.");
        return redirect()->back();
    }
}
