<?php
/**
 * CustomGroup Eloquent Model
 *
 * PHP Version 7
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\CustomFields;

use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Events\Admin\RegisterCustomGroupModels;

/**
 * CustomGroup Eloquent Model
 *
 * This is used to define a group in which multiple fields reside, letting the
 * system know of custom tables which have been created to store additional
 * data for an item/node.
 *
 * @category Mtc\Core\CustomFields
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class CustomGroup extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title'
    ];

    /**
     * Models that have said they would like to use this group, populated by a
     * registration event.
     *
     * @var array
     */
    protected static $models = [];

    /**
     * Retrieve the child fields for the group, ordered by 'order' ascending.
     *
     * @return IIlluminate\Database\Eloquent\Relations\HasMany
     */
    public function fields()
    {
        return $this->hasMany(CustomField::class, 'group_id')
            ->orderBy('order');
    }

    /**
     * Retrieve a list of any models which are able to use this group.
     *
     * @return array
     */
    public function getRegisteredModels()
    {
        if (empty(self::$models)) {
            event(new RegisterCustomGroupModels($this));
            sort(self::$models);
        }

        return self::$models;
    }

    /**
     * Register a model to CustomFields
     *
     * @param string $model_name Eloquent Model class name
     *
     * @return self
     */
    public function registerModel($model_name)
    {
        self::$models[] = $model_name;
        return $this;
    }

    /**
     * Retrieve the models which have been selected to use this group.
     *
     * @return IIlluminate\Database\Eloquent\Relations\HasMany
     */
    public function models()
    {
        return $this->hasMany(CustomGroupModel::class, 'group_id');
    }
}
