<?php
/**
 * Controller for Taxonomies
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Http\Controllers\Admin;

use Baum\MoveNotPossibleException;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Taxonomy;

/**
 * Resource Controller for Taxonomies
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package  Mtc\Core
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class TaxonomyController extends Controller
{
    /**
     * Ensure this is only accessible to those with the
     * 'manage-custom-field-taxonomies' permission.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:manage-custom-field-taxonomies');
    }
    /**
     * Display a listing of the resource.
     *
     * By default show the view, but if this is carried out by JSON show it
     * a list of taxonomies with their children.
     *
     * @return \Illuminate\View\View|Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        if ($request->wantsJson()) {
            $data = Taxonomy::roots()->get();

            return [
                'children' => $data->map(
                    function ($item) {
                        $item->isExpanded = false;
                        return $item;
                    }
                )
            ];
        }

        return view('core::admin.taxonomies.index');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Illuminate\Http\Request $request Incoming Request
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $taxonomy = Taxonomy::create(
            [
            'title' => $request->input('title')
            ]
        );

        return $this->index($request);
    }

    /**
     * Display the specified resource.
     *
     * @param Request  $request  Illuminate Request
     * @param Taxonomy $taxonomy Taxomony Model
     *
     * @return \Illuminate\Http\Response JSON data
     */
    public function show(Request $request, Taxonomy $taxonomy)
    {
        $taxonomy->children->map(
            function ($child) {
                // Set a default value for Vue to hook into.
                $child->isExpanded = false;
                return $child;
            }
        );
        return $taxonomy;
    }

    /**
     * Remove the specified resource from storage.
     *
     * If AJAX request, show a success message.
     *
     * @param Request  $request  Illuminate Request
     * @param Taxonomy $taxonomy Taxomony Model
     *
     * @return \Illuminate\Http\Response|void
     */
    public function destroy(Request $request, Taxonomy $taxonomy)
    {
        if ($taxonomy->node) {
            $taxonomy->node->delete();
        }

        $taxonomy->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }
    }

    /**
     * Create a child category.
     *
     * @param Request  $request  Request
     * @param Taxonomy $taxonomy Taxonomy Model
     *
     * @return \Illuminate\Http\Response JSON data
     */
    public function storeChild(Request $request, Taxonomy $taxonomy)
    {
        $child = $taxonomy->children()->create(
            [
            'title' => $request->input('title')
            ]
        );
        return $this->show($request, $taxonomy);
    }

    /**
     * List categories as ID => Title pair
     *
     * @param Request $request
     * @return mixed Collection
     */
    public function displayList(Request $request)
    {
        $taxonomies = Taxonomy::orderBy('lft')->get();

        // If there are taxonomies (and their children) to exclude
        if ($request->input('exclude', false)) {
            $exclude = Taxonomy::find($request->input('exclude', false));
            $taxonomies = $taxonomies->reject(function ($taxonomy) use ($exclude) {

                if ($taxonomy instanceof Taxonomy) {
                    return ($taxonomy->isDescendantOf($exclude) || $taxonomy->equals($exclude));
                }
            });
        }

        // Map out structure of ID => title
        $taxonomies = $taxonomies->keyBy('id')
            ->map(function ($taxonomy) {
                if (is_string($taxonomy)) {
                    return $taxonomy;
                }

                return str_repeat('-', $taxonomy->depth) . ' ' . $taxonomy->title;
            })->prepend(trans('core::fields.root_level'));

        // Return a prepared response
        // Doing this instead of collection as collection is already sorted and we don't want the order to change
        return response()->json($taxonomies);

    }

    /**
     * Update order of given taxonomies
     *
     * @param Request $request
     */
    public function updateOrder(Request $request)
    {
        /** @var Taxonomy $parent */
        $parent = Taxonomy::find($request->input('taxonomies.0.parent_id'));

        foreach ($request->input('taxonomies', []) as $key => $taxonomy_data) {
            /** @var Taxonomy $taxonomy */
            $taxonomy = Taxonomy::find($taxonomy_data['id']);

            try {
                if (!empty($previous_taxonomy)) {
                    $taxonomy->moveToRightOf($previous_taxonomy);
                } else {
                    $taxonomy->makeFirstChildOf($parent);
                }
            } catch (MoveNotPossibleException $exception) {
                // Ignore impossible move and proceed
                // e.g. the order of the first child has not changed causes the exception
            }

            // set this taxonomy as previous so we can put next one besides this one
            $previous_taxonomy = $taxonomy;
        }
    }

    /**
     * Update order of given taxonomies
     *
     * @param Request $request
     */
    public function changeParent(Request $request)
    {
        /** @var Taxonomy $taxonomy */
        $taxonomy = Taxonomy::find($request->input('taxonomy'));

        if ($request->input('new_parent') == 0) {
            $taxonomy->makeRoot();

        } else {
            $parent = Taxonomy::find($request->input('new_parent'));
            $taxonomy->makeChildOf($parent);
        }
    }
}
