<?php
/**
 * Class Seo\Defaults
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Models\Seo;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;

/**
 * Class Seo\Defaults
 *
 * Seo Defaults model
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Defaults extends Model
{

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'seo_defaults';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'node_type',
        'path',
        'title',
        'description'
    ];

    /**
     * @var string $taxonomy_tag_regex Regex to find Taxonomy tags and replace them
     */
    protected static $taxonomy_tag_regex = "#{TX:\[([a-zA-Z0-9 \-\_]+)\]:([A-Z]+)}#";

    /**
     * Get the seo defaults that are set for this node
     *
     * @param Node $node
     * @return Defaults|null
     */
    public static function getNodeDefaults(Node $node)
    {
        /**
         * @var self $default
         */
        $default = self::where('node', $node->nodeable_type)
            ->first();

        if ($default) {
            $default->convertTagsToValuesForNode($node);
        }
        return $default;
    }

    /**
     * Get the Seo Defaults that are set for this url
     *
     * Uses inverse matching (/browse/page-2 must match the entry for /browse)
     *
     * @param View $view view to render
     * @param $url
     * @return mixed
     */
    public static function getDefaultsFromUrl(View $view, $url)
    {
        $url = DB::getPdo()->quote($url);
        $default = self::where('path', '!=', '')
            ->whereRaw(DB::raw($url . ' LIKE CONCAT(`path`, "%")'))
            ->first();

        if (!$default) {
            $default = new self;
        }

        $view_data = $view->getData();

        // Get supported Tags
        $from = self::getSupportedReplaceTags()->toArray();
        ksort($from);

        // Start setting values for filling out placeholders
        $to = [
            'page' => !empty($view_data['page']) ? $view_data['page'] : 1,
            'site_name' => config('app.name'),
        ];


        if (!empty($view_data['selected'])) {
            $default->title = $default->matchTaxonomyTagsForUrl($view_data['selected'], $default->title);
            $default->description = $default->matchTaxonomyTagsForUrl($view_data['selected'], $default->description);
        }


        // Loop through all $from placeholders and replace ones that are not available with emptiness
        foreach ($from as $key => $value) {
            if (!isset($to[$key])) {
                $to[$key] = '';
            }
        }
        ksort($to);

        // convert tags to values
        $default->title = str_replace($from, $to, $default->title);
        $default->description = str_replace($from, $to, $default->description);


        return $default;
    }

    /**
     * Process taxonomy tags for url based pages
     *
     * @param array $selected_taxonomies list of selections
     * @param string $string_to_process meta tag to fill out with data
     * @return string processed string
     */
    private function matchTaxonomyTagsForUrl($selected_taxonomies, $string_to_process = '')
    {
        // Make sure we have a taxonomy tag in regex to process
        if (preg_match_all(self::$taxonomy_tag_regex, $string_to_process, $title_matches)) {
            // Selections come up as nodes instead of Taxonomy, so reload them
            $selected_taxonomies = Taxonomy::whereIn('id', $selected_taxonomies->pluck('id'))
                ->orderBy('depth', 'asc')
                ->get();

            // No selections, no info we can tweak
            if (count($selected_taxonomies) == 0) {
                return $string_to_process;
            }

            // $title_matches[1] stores the taxonomy name
            foreach ($title_matches[1] as $index => $taxonomy_title) {
                // $title_matches[2] stores the depth (root/current)
                $taxonomy_depth = $title_matches[2][$index];

                // Loop through all taxonomies
                foreach ($selected_taxonomies as $node_taxonomy) {
                    // Find root taxonomy
                    $root_taxonomy = $node_taxonomy->getRoot();

                    // If this is a taxonomy we need
                    if ($root_taxonomy->title == $taxonomy_title) {
                        // Check if we need to set parent or child value
                        if ($taxonomy_depth === 'ROOT') {
                            $string_to_process = str_replace(
                                $title_matches[0][$index],
                                    $root_taxonomy->title,
                                    $string_to_process
                            );
                        } else {
                            $string_to_process = str_replace(
                                $title_matches[0][$index],
                                    $node_taxonomy->title,
                                    $string_to_process
                            );
                        }
                    }
                }
            }
        }
        return $string_to_process;
    }

    /**
     * Process taxonomy tags for node based pages
     *
     * @param Node $node node of current page object
     * @param string $string_to_process meta tag to fill out with data
     * @return string processed string
     */
    private function matchTaxonomyTagsForNode(Node $node, $string_to_process = '')
    {
        // Make sure we have a taxonomy tag in regex to process
        if (preg_match_all(self::$taxonomy_tag_regex, $string_to_process, $title_matches)) {
            // $title_matches[1] stores the taxonomy name
            foreach ($title_matches[1] as $index => $taxonomy_title) {
                // $title_matches[2] stores the depth (root/current)
                $taxonomy_depth = $title_matches[2][$index];

                // Fetch all nodes in ascending depth (so current depth is last)
                $node_taxonomies = $node->taxonomies()
                    ->orderBy('depth', 'asc')
                    ->get();

                // Loop through all taxonomies
                foreach ($node_taxonomies as $node_taxonomy) {
                    // Find root taxonomy
                    $root_taxonomy = $node_taxonomy->getRoot();

                    // If this is a taxonomy we need
                    if ($root_taxonomy->title == $taxonomy_title) {
                        // Check if we need to set parent or child value
                        if ($taxonomy_depth === 'ROOT') {
                            $string_to_process = str_replace(
                                $title_matches[0][$index],
                                    $root_taxonomy->title,
                                    $string_to_process
                            );
                        } else {
                            $string_to_process = str_replace(
                                $title_matches[0][$index],
                                    $node_taxonomy->title,
                                    $string_to_process
                            );
                        }
                    }
                }
            }
        }
        return $string_to_process;
    }

    /**
     * Fill the default values with real values from node
     *
     * @param Node|false $node Node for current page
     * @param string $url url of this page
     */
    protected function convertTagsToValuesForNode($node, $url = '')
    {
        // Get supported Tags
        $from = self::getSupportedReplaceTags()->toArray();
        ksort($from);

        // Start setting values for filling out placeholders
        $to = [
            'title' => $node->title,
            'description' => $node->description,
            'site_name' => config('app.name'),
        ];

        $this->title = $this->matchTaxonomyTagsForNode($node, $this->title);
        $this->description = $this->matchTaxonomyTagsForNode($node, $this->description);


        // Loop through all $from placeholders and replace ones that are not available with emptiness
        foreach ($from as $key => $value) {
            if (!isset($to[$key])) {
                $to[$key] = '';
            }
        }
        ksort($to);

        // convert tags to values
        $this->title = str_replace($from, $to, $this->title);
        $this->description = str_replace($from, $to, $this->description);
    }

    /**
     * Set up a list of all available SEO Tags
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getSupportedReplaceTags()
    {
        $tags = collect([
            'title' => '{TITLE}',
            'description' => '{DESCRIPTION}',
            'site_name' => '{SITE_NAME}',
            'page' => '{PAGE}'
        ]);

        // Loop through Taxonomies
        Taxonomy::roots()
            ->get()
            ->each(function ($taxonomy) use ($tags) {
                $tags->put('txr_' . $taxonomy->id, '{TX:[' . $taxonomy->title . ']:ROOT}');
                $tags->put('txc_' . $taxonomy->id, '{TX:[' . $taxonomy->title . ']:CURRENT}');
            });

        return $tags;
    }
}
