<?php
/**
 * Class RoyalMailAddressFinder
 *
 * @package Mtc/Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Core\AddressFinder\Drivers;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Mtc\Core\Contracts\AddressFinder;
use Mtc\Core\Traits\AddressValidator;

/**
 * Class RoyalMailAddressFinder
 *
 * Class that implements address searching through Royal Mail Database
 * This database is Located on paf.mtcassets.com and can be accessed directly by mysql.
 *
 * @package Mtc/Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class RoyalMailAddressFinder implements AddressFinder
{
    /**
     * Implement search functionality on Royal Mail
     *
     * @param string $query given search string - either postcode or partial address
     * @return Collection Collection of found addresses
     * @throws \Exception Exception about wildcard search
     */
    public function search($query)
    {
        if (!config('core.address_finder.wildcard_term_search')) {
            return $this->searchByPostCode($query);
        }

        throw new \Exception("'Wildcard search is not implemented");
        // Todo: implement wildcard search
    }

    /**
     * Search explicitly by postcode
     *
     * @param string $query search query (postcode to find)
     * @return Collection|mixed collection of items to display
     * @throws \Exception error about invalid postcode
     */
    public function searchByPostCode($query)
    {
        // Normalize the postcode to the format we are safe to search
        $postcode = AddressValidator::normalizePostcode($query, 'GB');

        $postcode_validator = Validator::make([
            'postcode' => $postcode
        ], [
            'postcode' => 'postcode:GB'
        ]);

        // Throw an error if the given string is not a valid postcode
        if ($postcode_validator->fails()) {
            throw new \Exception("'{$query}' is not a valid UK postcode");
        }

        /** @var Collection $address_collection */
        $address_collection = Cache::remember("royal_mail_search:{$postcode}", 60, function () use ($postcode) {
            return DB::connection('paf_mtcassets')
                ->table('Address_Complete')
                ->where('postcode', 'like', "{$postcode}%")
                ->get();
        });

        // If we have results returned we need to map the values to a readable format
        if ($address_collection->count() > 0) {
            return $this->prepareForDisplay($address_collection, $query);
        }

        return $address_collection;
    }

    /**
     * Convert the collection of addresses to a render friendly format
     *
     * @param Collection $address_collection
     * @param string $query_postcode original postcode value to show search visibility
     * @return mixed
     */
    private function prepareForDisplay($address_collection, $query_postcode)
    {
        return $address_collection->map(function ($address) use ($query_postcode) {
            // Build up the values for address 1 based on full details
            $address1 = implode(' ', [
                $address->{'Department Name'},
                $address->{'Organisation Name'},
                !empty($address->{'PO Box Number'}) ? 'POBox ' . $address->{'PO Box Number'} : '',
                !empty($address->{'Building Number'}) ? $address->{'Building Number'} : '',
                $address->{'Sub Building Name'},
                $address->{'Building Name'},

            ]);

            // Build up the values for address 2 based on full details
            $address2 = implode(' ', [
                trim($address->{'Thoroughfare Name'}),
                trim($address->{'Thoroughfare Descriptor'}),
                trim($address->{'Dependent Thoroughfare Name'}),
                trim($address->{'Dependent Thoroughfare Descriptor'}),

            ]);

            // Build up the address displayed for user
            // This contains postcode in query to ensure that we can search results
            $display_address = implode(', ', [
                ucwords(strtolower(trim($address1))),
                ucwords(strtolower(trim($address2))),
                ucwords(strtolower($address->{'Post Town'})),
                strtoupper($query_postcode)
            ]);

            return collect([
                'id' => $address->{'Address Key'},
                'display_address' => $display_address,
                'address1' => trim($address1),
                'address2' => trim($address2),
                'city' => $address->{'Post Town'},
                'state' => '',
                'postcode' => $address->Postcode,
                // Since this is a Royal mail DB all addresses are GB
                'country' => 'GB',

            ]);
        });
    }
}