<?php
/**
 * Class GenerateSitemap
 *
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @version 02.07.2017
 */
namespace Mtc\Core\Console\Commands;

use Illuminate\Console\Command;
use Carbon\Carbon;
use Mtc\Core\Node;
use Illuminate\Support\Collection;
use Mtc\Core\Events\SitemapGenerated;
use Spatie\Sitemap\Sitemap;
use Spatie\Sitemap\Tags\Url;
use Spatie\Sitemap\SitemapGenerator;

/**
 * Class GenerateSitemap
 *
 * This script generates sitemap
 *
 * @package Mtc\Core
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @version 02.07.2017
 */
class GenerateSitemap extends Command
{
    /**
     * The console command name.
     *
     * @var string
     */
    protected $signature = 'sitemap:generate';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generates the sitemap';

    /**
     * @var Sitemap
     */
    protected $sitemap;

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        $this->sitemap = SitemapGenerator::create(config('app.url'))
            ->getSitemap();

        $this->collectNodes()
            ->map(function ($url) {
                $this->addUrl($url);
            });

        event(new SitemapGenerated($this));

        $this->sitemap->writeToFile(public_path('sitemap.xml'));

    }

    /**
     * Function collectNodes()
     *
     * We need this because sitemap crawler wont pick up products as they're loaded dynamically.
     *
     * @return Collection
     */
    protected function collectNodes() : Collection
    {
        return Node::query()
            ->where('visibility', 'public')
            ->where('url', '!=', '')
            ->pluck('url');
    }

    /**
     * Function addUrl()
     *
     * Adds url to sitemap. Checks if not added already first
     *
     * @param string $url
     *
     */
    public function addUrl(string $url)
    {
        $url = config('app.url') . $url;
        if (!$this->sitemap->hasUrl($url)) {
            $this->sitemap->add(Url::create($url)
                ->setLastModificationDate(Carbon::yesterday())
                ->setChangeFrequency(Url::CHANGE_FREQUENCY_DAILY)
                ->setPriority(0.8));
        }

    }
}