<?php
/**
 * Core Country Class
 *
 * PHP Version 7
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Rihards Silins <rihards.silins@mtcmedia.co.uk>
 */

namespace Mtc\Core;

use Illuminate\Support\Collection;

/**
 * Eloquent Model for Countries
 *
 * @category Mtc\Core
 * @package  Mtc\Core
 * @author   Rihards Silins Rihards Silins <rihards.silins@mtcmedia.co.uk>
 * @author   Vladislavs Ignatjevs <vladislavs.ignatjevs@mtcmedia.co.uk>
 */
class Country extends \Illuminate\Database\Eloquent\Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'code',
        'status',
        'order',
    ];

    /**
     * @var string[] $hidden List of attributes we don't show when converting to public viewing
     */
    protected $hidden = [
        'id',
        'order',
        'created_at',
        'updated_at',
        'postcode_regex'
    ];

    /**
     * A relation to the states for this country.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function states()
    {
        return $this->hasMany(CountryState::class);
    }

    /**
     * Get a list of active countries
     * Countries are keyed by their code
     * and ordered by index with default country placed at top
     *
     * @return Collection
     */
    public static function getOrderedActiveCountries()
    {
        return self::whereStatus(1)
            ->with('states')
            ->get()
            ->keyBy(function (self $item) {
                return $item->code;
            })
            ->map(function (self $country) {
                $country->has_states = count($country->states) > 0;
                return $country;
            })
            ->sortBy(function (self $item, $index) {
                return $item->code == config('core.default_country', 'GB') ? -1 : $index;
            });
    }

    /**
     * Find the country name for the code.
     *
     * Falls back to returning the country code if the code was not found.
     *
     * @param string $code country code
     * @return string country code name
     */
    public static function getNameFromCode($code)
    {
        return self::where('code', $code)->first()->name ?? $code;
    }
}
