<?php
/**
 * Core Event Service Provider
 *
 * PHP Version 7
 *
 * @category Mtc\Core\Providers
 * @package  Mtc\Core\Admin
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Core\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Event;
use Mtc\Core\Admin\Menu;
use Mtc\Core\Events\Admin\MenuLoading;
use Mtc\Core\Events\HttpNotFoundEvent;
use Mtc\Core\Listeners\BespokeUrlMatcher;
use Mtc\Core\Node;

/**
 * Within this class are all the events which the Core is listening and
 * reacting to.
 *
 * @category Mtc\Core\Providers
 * @package  Mtc\Core\Admin
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class EventServiceProvider extends ServiceProvider
{
    /**
     * Register any events for your application.
     *
     * @return void
     */
    public function boot()
    {
        $this->registerAdminMenuEvents();

        Event::listen(
            'Illuminate\Auth\Events\Attempting',
            'Mtc\Core\Listeners\ValidateMtcLogin'
        );
        Event::listen('eloquent.deleted:*', 'Mtc\Core\Listeners\DeleteNode');
        Event::listen(
            'eloquent.saving: ' . Node::class,
            'Mtc\Core\Listeners\DeleteCustomData'
        );
        Event::listen(
            'eloquent.saved: ' . Node::class,
            'Mtc\Core\Listeners\SaveCustomData'
        );

        // Check for Not found events to look if we have bespoke url set
        Event::listen(
            HttpNotFoundEvent::class,
            BespokeUrlMatcher::class
        );

        Event::listen(
            'members.dashboard',
            function () {
                return [
                    'url' => route('core.member.details.index'),
                    'icon' => 'fa-user',
                    'title' => trans('core::fields.account_details'),
                    'subtitle' => trans('core::fields.edit_now'),
                ];
            }
        );
    }

    /**
     * Register all events related to admin menu
     */
    private function registerAdminMenuEvents()
    {
        Event::listen(
            MenuLoading::class, function () {

            // Management section
            Menu::addSection(
                'Content',
                'content'
            );

            // Management section
            Menu::addSection(
                'Management',
                'management'
            );

            // General management - users, permissions, currencies, languages etc.
            Menu::addMenu(
                'management',
                'General',
                '',
                'general',
                'glyphicon glyphicon-wrench'
            );

            Menu::addSubMenu(
                'general',
                'Manage Users',
                'manage-users',
                route('core.admin.users.index'),
                'manage_users'
            );

            Menu::addSubMenu(
                'general',
                'Manage Roles',
                'manage-roles',
                route('core.admin.roles.index'),
                'manage_roles'
            );

            if (config('core.currencies_enabled', false)) {
                Menu::addSubMenu(
                    'general',
                    'Manage Currencies',
                    'manage-currencies',
                    action('\\Mtc\Core\Http\Controllers\Admin\CurrencyController@index'),
                    'manage_currencies'
                );
            }


            Menu::addMenu(
                'management',
                'Custom Fields',
                'manage-custom-fields',
                'custom-fields',
                'glyphicon glyphicon-log-in'
            );
            Menu::addSubMenu(
                'custom-fields',
                'Groups',
                'manage-custom-field-groups',
                route('core.admin.custom-fields.groups.index'),
                'custom-field-groups'
            );
            Menu::addSubMenu(
                'custom-fields',
                'Taxonomies',
                'manage-custom-field-taxonomies',
                route('core.admin.taxonomies.index'),
                'custom-field-taxonomies'
            );

            // SEO menu items
            Menu::addMenu(
                'management',
                trans('Seo'),
                'manage-seo',
                'seo-admin',
                'glyphicon glyphicon-eye-open'
            );
            Menu::addSubMenu(
                'seo-admin',
                trans('Seo Settings'),
                'manage-seo',
                route('core.admin.seo.settings.index'),
                'manage-seo-settings'
            );
            Menu::addSubMenu(
                'seo-admin',
                trans('Custom URLs'),
                'manage-seo',
                route('core.admin.seo.urls.index'),
                'manage-urls'
            );
            Menu::addSubMenu(
                'seo-admin',
                trans('Listing Page Content'),
                'manage-seo',
                route('core.admin.seo.browse_content.index'),
                'manage-seo-browse-content'
            );
            Menu::addSubMenu(
                'seo-admin',
                trans('Seo Defaults'),
                'manage-seo-defaults',
                route('core.admin.seo.defaults.index'),
                'manage-seo-defaults'
            );
            Menu::addSubMenu(
                'seo-admin',
                trans('Page Metadata'),
                'manage-seo',
                route('core.admin.seo.page_meta_tags.index'),
                'manage-seo-pages'
            );
            Menu::addSubMenu(
                'seo-admin',
                trans('Knowledge Graph'),
                'manage-seo',
                route('core.admin.seo.knowledge_graph.index'),
                'manage-seo'
            );

        }
        );

    }
}
