<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Mtc\Core\Admin\Builder;
use Mtc\Core\EmailTemplate;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Mail\TemplateEmail;

/**
 * Control admin actions over email templates.
 *
 * @category Mtc\Core\Http\Controllers\Admin
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class EmailController extends Controller
{
    /**
     * Ensure this is only accessible to those with the 'manage-roles'
     * permission.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:manage-emails');
        $this->middleware('permission:admin-email-templates', [
            'only' => [
                'create',
                'store',
                'delete'
            ]
        ]);
    }

    /**
     * Render a browse page showing all email templates available in the system.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return (new Builder('core.admin.emails', EmailTemplate::query()))
            ->columns([
                'title' => trans('fields.title'),
                'active' => trans('core::fields.active'),
            ])
            ->setAddPermission('admin-email-templates')
            ->setDeletePermission('admin-email-templates')
            ->data([
                'active' => function ($item) {
                    return $item->active ? __('core::fields.yes') : __('core::fields.no');
                }
            ])
            ->view();
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return (new ItemBuilder('core.admin.emails', new EmailTemplate()))
            ->view('core::admin.emails.form', [
                'title' => trans('core::fields.user_role')
            ]);
    }

    /**
     * Create a new Email Template
     *
     * @param Request $request
     * @param EmailTemplate $email
     * @return \Illuminate\Http\Response Redirect to edit page with success message
     */
    public function store(Request $request, EmailTemplate $email)
    {
        $this->validate($request, [
            'item.title' => 'required',
        ]);

        $email->fill($request->input('item'))
            ->save();

        $request->session()->flash('success', "Email Template '{$email->title}' has been saved.");
        return redirect(route('core.admin.emails.edit', ['email' => $email->id]));
    }

    /**
     * Render preview of the email template
     *
     * @param EmailTemplate $email
     * @return \Illuminate\View\View
     */
    public function show(EmailTemplate $email)
    {
        if (app()->bound('debugbar')) {
            app('debugbar')->disable();
        }

        if ($email->layout) {
            $data = [
                'email' => $email
            ];

            if ($email->model) {
                $data = array_merge($data, $email->model->getEmailTemplatePreviewData());
            } elseif (!empty($email->model_type)) {
                $data = array_merge($data, (new $email->model_type)->getEmailTemplatePreviewData());
            }
            return (new TemplateEmail($email->title, $email->layout, $data))->render();
        }
    }

    /**
     * Show a view to allow the admin to edit an existing email template.
     *
     * @param EmailTemplate $email
     * @return \Illuminate\View\View
     */
    public function edit(EmailTemplate $email)
    {
        return (new ItemBuilder('core.admin.emails', $email))
            ->view('core::admin.emails.form', [
                'title' => trans('core::fields.email_template')
            ]);
    }

    /**
     * Save an existing email template.
     *
     * @param Request $request
     * @param EmailTemplate $email
     * @return \Illuminate\Http\Response Redirect to role browse
     */
    public function update(Request $request, EmailTemplate $email)
    {
        return $this->store($request, $email);
    }

    /**
     * Delete an existing email template. If AJAX, we will send the OK. If not, we flash
     * a success message.
     *
     * @param Request $request Normal request
     * @param EmailTemplate $email
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, EmailTemplate $email)
    {
        $email->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Email Template '{$email->title}' has been deleted.");
        return redirect()->back();
    }
}
